(ns antistock.config.schema
  (:require [schema.core :as s]))

(s/defschema Api
  "The schema for the API service."
  {:bind-address s/Str
   :bind-port s/Int})

(s/defschema ApiClient
  "The schema for the API client."
  {:scheme (s/enum :http :https)
   :server-name s/Str
   :server-port s/Int})

(s/defschema Aws
  "The schema for the AWS service."
  {:access-key s/Str
   :account-number s/Str
   :secret-key s/Str})

(s/defschema Broker
  "The schema for the broker service."
  {:password s/Str
   :scheme (s/enum :amqp)
   :server-name s/Str
   :server-port s/Int
   :username s/Str
   :vhost s/Str})

(s/defschema Curator
  "The schema for the Curator service."
  {:password s/Str
   :server-name s/Str
   :server-port s/Int
   :username s/Str})

(s/defschema Db
  "The schema for the database service."
  {:name s/Str
   :password s/Str
   :scheme (s/enum :postgresql)
   :server-name s/Str
   :server-port s/Int
   :username s/Str
   (s/optional-key :pool) (s/enum :bonecp :c3p0 :hikaricp nil)
   (s/optional-key :test?) s/Bool})

(s/defschema Env
  "The schema for the environment."
  {s/Keyword s/Str})

(s/defschema Reload
  "The schema for the namespace reload service."
  {:disabled s/Bool
   :source-paths [s/Str]})

(s/defschema Twitter
  "The schema for the Twitter client."
  {:consumer-key s/Str
   :consumer-secret s/Str
   :access-token s/Str
   :access-token-secret s/Str})

(s/defschema TwitterWorker
  "The schema for the Twitter worker."
  {:broker Broker
   :exchange s/Str
   :prefetch-count s/Int
   :queue s/Str
   :routing-key s/Str})

(s/defschema Web
  "The schema for the web service."
  {:bind-address s/Str
   :bind-port s/Int})

(s/defschema Spark
  "The schema for Spark applications."
  {:app-name s/Str
   :master s/Str})

(s/defschema WebClient
  "The schema for the web client."
  {:scheme (s/enum :http :https)
   :server-name s/Str
   :server-port s/Int})

(s/defschema WikipediaViewsWorker
  "The schema for the Wikipedia views worker."
  {:broker Broker
   :exchange s/Str
   :prefetch-count s/Int
   :queue s/Str
   :routing-key s/Str})

(s/defschema Zookeeper
  "The schema for the Zookeeper service."
  {:password s/Str
   :server-name s/Str
   :server-port s/Int
   :username s/Str})

;; System Schemas

(s/defschema ApiSystem
  "The schema for worker system."
  {:api Api
   :api-client ApiClient
   :broker Broker
   :db Db})

(s/defschema WorkerSystem
  "The schema for worker system."
  {:broker Broker
   :db Db
   :twitter TwitterWorker
   :wikipedia-views WikipediaViewsWorker})

(s/defschema Config
  "The schema for the config."
  {:api Api
   :api-client ApiClient
   :aws Aws
   :broker Broker
   :curator Curator
   :db Db
   :reload Reload
   :twitter Twitter
   :web Web
   :web-client WebClient
   :zookeeper Zookeeper})
