(ns antistock.config.core
  (:require [antistock.config.schema :as schemas]
            [schema.core :as s]
            [clojure.edn :as edn]))

(s/defn ^:always-validate api :- schemas/Api
  "Return the API config from `env`."
  [env]
  {:bind-address (:as-api-bind-address env)
   :bind-port (or (edn/read-string (:as-api-bind-port env))
                  (edn/read-string (:port env)))})

(s/defn ^:always-validate api-client :- schemas/ApiClient
  "Return the API client config from `env`."
  [env]
  {:scheme (keyword (:as-api-scheme env))
   :server-name (:as-api-server-name env)
   :server-port (edn/read-string (:as-api-server-port env))})

(s/defn ^:always-validate aws :- schemas/Aws
  "Return the AWS config from `env`."
  [env]
  {:access-key (:as-aws-access-key env)
   :account-number (:as-aws-account-number env)
   :secret-key (:as-aws-secret-key env)})

(s/defn ^:always-validate broker :- schemas/Broker
  "Return the broker config from `env`."
  [env]
  {:password (:as-broker-password env)
   :scheme (or (:as-broker-scheme env) :amqp)
   :server-name (:as-broker-server-name env)
   :server-port (edn/read-string (:as-broker-server-port env))
   :username (:as-broker-username env)
   :vhost (or (:as-broker-vhost env) "/")})

(s/defn ^:always-validate db :- schemas/Db
  "Return the database config from `env`."
  [env]
  {:name (:as-db-name env)
   :password (:as-db-password env)
   :pool (some-> env :as-db-pool keyword)
   :test? (true? (edn/read-string (:as-db-test env)))
   :scheme (keyword (:as-db-scheme env))
   :server-name (:as-db-server-name env)
   :server-port (edn/read-string (:as-db-server-port env))
   :user (:as-db-username env)})

(s/defn ^:always-validate twitter :- schemas/Twitter
  "Return the Twitter config from `env`."
  [env]
  {:consumer-key (:as-twitter-consumer-key env)
   :consumer-secret (:as-twitter-consumer-secret env)
   :access-token (:as-twitter-access-token env)
   :access-token-secret (:as-twitter-access-token-secret env)})

(s/defn ^:always-validate reload :- schemas/Reload
  "Return the namespace reload config from `env`."
  [env]
  {:disabled (true? (edn/read-string (:as-reload-disabled env)))
   :source-paths ["src" "test"]})

(s/defn ^:always-validate spark :- schemas/Spark
  "Return the Spark application config."
  [env]
  {:app-name (or (:as-spark-app-name env) "antistock")
   :master (or (:as-spark-master env) "local")})

(s/defn ^:always-validate web :- schemas/Web
  "Return the web config from `env`."
  [env]
  {:bind-address (:as-web-bind-address env)
   :bind-port (edn/read-string (:as-web-bind-port env))})

(s/defn ^:always-validate web-client :- schemas/WebClient
  "Return the web client config from `env`."
  [env]
  {:scheme (keyword (:as-web-scheme env))
   :server-name (:as-web-server-name env)
   :server-port (edn/read-string (:as-web-server-port env))})

(s/defn ^:always-validate zookeeper :- schemas/Zookeeper
  "Return the zookeeper config from `env`."
  [env]
  {:password (:as-zookeeper-password env)
   :server-name (:as-zookeeper-server-name env)
   :server-port (edn/read-string (:as-zookeeper-server-port env))
   :username (:as-zookeeper-username env)})

(s/defn ^:always-validate config :- schemas/Config
  "Return the system config from `env`."
  [env]
  {:api (api env)
   :api-client (api-client env)
   :aws (aws env)
   :broker (broker env)
   :db (db env)
   :reload (reload env)
   :twitter (twitter env)
   :web (web env)
   :web-client (web-client env)
   :zookeeper (zookeeper env)})
