(ns antistock.cli.nasdaq
  (:refer-clojure :exclude [distinct group-by update])
  (:require [antistock.config.core :refer [config]]
            [antistock.core :refer [safe-trim]]
            [antistock.db :as db]
            [antistock.db.system :refer [new-db]]
            [antistock.system :as system]
            [clojure.data.csv :refer [read-csv]]
            [clojure.java.io :refer [reader]]
            [clojure.tools.logging :as log]
            [commandline.core :refer [print-help with-commandline]]
            [com.stuartsierra.component :as component]
            [datumbazo.core :refer :all :exclude [new-db run]]
            [environ.core :refer [env]])
  (:gen-class))

(def csv-header
  [:symbol :name :last-sale :market-cap :adr-tso
   :ipo-year :sector :industry :summary-quote])

(defn create-import-table
  "Create a temporary table to import NASDAQ company data."
  [db]
  @(create-table db :nasdaq
     (column :exchange-id :integer)
     (column :symbol :text)
     (column :name :text)
     (column :last-sale :text)
     (column :market-cap :text)
     (column :adr-tso :text)
     (column :ipo-year :text)
     (column :sector :text)
     (column :industry :text)
     (column :summary-quote :text)
     (temporary true)
     (if-not-exists true)))

(defn clear-import-table
  "Delete all rows from the temporary table."
  [db] @(delete db :nasdaq))

(defn companies-url
  "Returns the NASDAQ company data url for `exchange`."
  [exchange]
  (format (str "http://www.nasdaq.com/screening/companies-by-name.aspx?"
               "letter=0&exchange=%s&render=download")
          (case (:id exchange)
            1 "nyse"
            2 "nasdaq"
            3 "amex"
            nil)))

(defn companies-by-url
  "Returns the NASDAQ company data at `url`."
  [url]
  (with-open [reader (reader url)]
    (doall (map #(zipmap csv-header %1)
                (rest (read-csv reader))))))

(defn companies-by-exchange
  "Returns the NASDAQ company data for `exchange`."
  [exchange]
  (when-let [url (companies-url exchange)]
    (map #(-> (assoc %1 :exchange-id (:id exchange))
              (update-in [:industry] safe-trim)
              (update-in [:name] safe-trim)
              (update-in [:sector] safe-trim)
              (update-in [:symbol] safe-trim)
              (update-in [:summary-quote] safe-trim))
         (companies-by-url url))))

(defn insert-sectors
  "Insert the sectors from the temporary table."
  [db]
  (insert db :sectors [:name]
    (select db (distinct [:nasdaq.sector])
      (from :nasdaq)
      (join :sectors '(on (= (lower :sectors.name)
                             (lower :nasdaq.sector)))
            :type :left)
      (where `(and (is-null :sectors.name)
                   (~(keyword "!~*") :nasdaq.sector "n/a"))))))

(defn insert-industries
  "Insert the industries from the temporary table."
  [db]
  (insert db :industries [:name]
    (select db (distinct [:nasdaq.industry])
      (from :nasdaq)
      (join :industries '(on (= (lower :industries.name)
                                (lower :nasdaq.industry)))
            :type :left)
      (where `(and (is-null :industries.name)
                   (~(keyword "!~*") :nasdaq.industry "n/a"))))))

(defn insert-companies
  "Insert the companies from the temporary table."
  [db]
  (insert db :companies [:sector-id :industry-id :name]
    (select db (distinct [:sectors.id :industries.id :nasdaq.name]
                         :on [:nasdaq.name])
      (from :nasdaq)
      (join :sectors '(on (= (lower :sectors.name)
                             (lower :nasdaq.sector)))
            :type :left)
      (join :industries '(on (= (lower :industries.name)
                                (lower :nasdaq.industry)))
            :type :left)
      (join :companies '(on (= (lower :companies.name)
                               (lower :nasdaq.name)))
            :type :left)
      (where `(and (is-null :companies.name)
                   (~(keyword "!~*") :nasdaq.name "n/a")))
      (order-by (nulls :nasdaq.name :last)))))

(defn insert-quotes
  "Insert the quotes from the temporary table."
  [db]
  (insert db :quotes [:symbol :exchange-id :company-id]
    (select db (distinct [:nasdaq.symbol :nasdaq.exchange-id :companies.id]
                         :on [:nasdaq.symbol])
      (from :nasdaq)
      (join :companies '(on (= (lower :companies.name)
                               (lower :nasdaq.name)))
            :type :left)
      (join :quotes '(on (and (= (lower :quotes.symbol)
                                 (lower :nasdaq.symbol))))
            :type :left)
      (where `(and (is-null :quotes.symbol))))))

(defn update-quotes
  "Update the quotes from the temporary table."
  [db]
  (update db :quotes {:exchange-id :u.exchange-id
                      :company-id :u.company-id}
          (from (as (select db (distinct [:nasdaq.symbol :nasdaq.exchange-id
                                          (as :companies.id :company-id)]
                                         :on [:nasdaq.symbol])
                      (from :nasdaq)
                      (join :companies
                            '(on (= (lower :companies.name)
                                    (lower :nasdaq.name)))))
                    :u))
          (where '(and (= (lower :quotes.symbol) (lower :u.symbol))
                       (= :quotes.exchange-id :u.exchange-id)))))

(defn load-data [db exchange]
  (let [companies (companies-by-exchange exchange)]
    (if (empty? companies)
      (log/warnf "No companies found for %s." (:name exchange))
      (do (log/infof "Importing stock symbols from %s."
                     (:name exchange))
          (create-import-table db)
          (clear-import-table db)
          @(insert db :nasdaq [] (values companies))))))

(defn import-from-exchange
  "Import sectors, industries and companies from `exchange`."
  [db exchange]
  (with-transaction [db db]
    (load-data db exchange)
    @(insert-sectors db)
    @(insert-industries db)
    @(insert-companies db)
    @(update-quotes db)
    @(insert-quotes db)))

(defn import-from-exchanges
  "Import sectors, industries and companies from `exchanges`."
  [db exchanges & [opts]]
  (doseq [exchange exchanges]
    (import-from-exchange db exchange)))

(defn exchanges
  "Return the exchanges to import from."
  [db args]
  (if (empty? args)
    (db/exchanges db)
    (mapcat #(deref (db/exchanges-like-name db %1)) args)))

(defn new-system
  [{:keys [db]}]
  (-> (component/system-map
       :db (new-db db))
      (component/system-using
       {:db []})))

(defn run [config args]
  (system/with-component [system (new-system config)]
    (with-connection [db (:db system)]
      (import-from-exchanges db (exchanges db args)))))

(defn -main [& args]
  (with-commandline [[opts args] args]
    [[h help "Print this help."]]
    (when (:help opts)
      (print-help "as scheduler")
      (System/exit 0))
    (run (config env) args)))
