(ns antistock.cli.wikipedia.views
  (:require [antistock.config.core :as config]
            [antistock.db.quotes :as quotes]
            [antistock.db.system :refer [with-db]]
            [antistock.queue :as broker]
            [antistock.time :as time]
            [datumbazo.core :refer [with-connection]]
            [clojure.tools.logging :as log]
            [commandline.core :refer [print-help with-commandline]]
            [environ.core :refer [env]]))

(defn run [config & [opts]]
  (with-db [db (:db config)]
    (with-connection [db db]
      (broker/with-broker [broker (:broker config)]
        (broker/with-channel [channel broker]
          (let [months (time/monthly-date-range (:start opts) (:end opts))]
            (doseq [quote (quotes/quotes db), month months]
              (broker/publish-edn
               channel "wikipedia" "wikipedia.page.views"
               {:quote quote :time month}
               {:persistent true}))
            (log/infof "Updating Wikipedia page views from %s until %s."
                       (time/year-month-day (first months))
                       (time/year-month-day (last months)))))))))

(defn -main
  [& args]
  (with-commandline [[opts args] args]
    [[e end "Load Wikipedia page views until END time." :time "END"]
     [h help "Print this help."]
     [s start "Load Wikipedia page views from START time." :time "START"]]
    (when (:help opts)
      (print-help "wikipedia views [OPTION...]")
      (System/exit 0))
    (run {:broker (config/broker env)
          :db (config/db env)}
      opts)))
