(ns antistock.broker
  (:require [clj-time.core :refer [now]]
            [antistock.db :as db]
            [antistock.portfolio :as portfolio]
            [antistock.order :refer [buy-order sell-order]]))

(defrecord Broker [name fee])

(def trade-king (->Broker "Trade King" 4.95))

(defn charge-transaction
  "Subtract the broker's transaction fee from `portfolio`."
  [broker portfolio]
  (-> (update-in portfolio [:money] - (:fee broker))
      (update-in [:broker-fee] #(+ (or %1 0) (:fee broker)))))

(defmulti apply-order
  "Apply `order` via `broker` to `portfolio`."
  (fn [broker portfolio order]
    (:side order)))

(defmethod apply-order :buy [broker portfolio order]
  (->> (portfolio/apply-order portfolio order)
       (charge-transaction broker)))

(defmethod apply-order :sell [broker portfolio order]
  (->> (portfolio/apply-order portfolio order)
       (charge-transaction broker)))

(defn buy-shares
  "Buy `amount` shares of `symbol` at `price` into portfolio."
  [broker portfolio symbol amount price]
  (apply-order broker portfolio (buy-order symbol amount price)))

(defn sell-shares
  "Sell `amount` shares of `symbol` at `price` into portfolio."
  [broker portfolio symbol amount price]
  (apply-order broker portfolio (sell-order symbol amount price)))
