(ns antistock.portfolio
  (:require [clojure.string :refer [upper-case]]
            [antistock.order :refer [buy-order sell-order]]))

(defrecord Portfolio [money shares prices])

(defn amount-of-shares
  "Returns the amount of shares of `symbol` in `portfolio`."
  [portfolio symbol] (get-in portfolio [:shares (upper-case symbol)]))

(defmulti apply-order
  "Apply `order` to `portfolio`."
  (fn [portfolio order] (:side order)))

(defmethod apply-order :buy [portfolio order]
  (let [{:keys [symbol amount price]} order
        current (amount-of-shares portfolio symbol)]
    (cond
      (neg? (- (:money portfolio) (* amount price)))
      (throw (ex-info
              (format "Can't buy %s shares of %s: No enough money left."
                      amount symbol)
              {:amount amount
               :symbol symbol}))
      :else
      (-> (update-in portfolio [:money] #(- %1 (* amount price)))
          (update-in [:shares symbol] #(+ (or %1 0) amount))
          (assoc-in [:prices symbol] price)))))

(defmethod apply-order :sell [portfolio order]
  (let [{:keys [symbol amount price]} order
        current (amount-of-shares portfolio symbol)]
    (cond
      (nil? current)
      (throw (ex-info
              (format "Can't sell %s shares of %s not in portfolio."
                      amount symbol)
              {:amount amount
               :symbol symbol}))
      (neg? (- current amount))
      (throw (ex-info
              (format "Can't sell more than %s shares of %s."
                      current symbol)
              {:current current
               :symbol symbol}))
      :else
      (-> (update-in portfolio [:money] #(+ %1 (* amount price)))
          (update-in [:shares symbol] #(- %1 amount))
          (assoc-in [:prices symbol] price)
          (cond-> (zero? (- current amount))
            (assoc :shares (dissoc (:shares portfolio) symbol)))
          (cond-> (zero? (- current amount))
            (assoc :prices (dissoc (:prices portfolio) symbol)))))))

(defn make-portfolio
  "Make an empty portfolio with `money`."
  [money] (->Portfolio money {} {}))

(defn buy-shares
  "Buy `amount` shares of `symbol` at `price` into portfolio."
  [portfolio symbol amount price]
  (apply-order portfolio (buy-order symbol amount price)))

(defn sell-shares
  "Sell `amount` shares of `symbol` at `price` into portfolio."
  [portfolio symbol amount price]
  (apply-order portfolio (sell-order symbol amount price)))
