(ns antistock.cli.features
  (:require [antistock.config.core :as config]
            [antistock.db.features :as features]
            [antistock.db.quotes :as quotes]
            [antistock.db.system :refer [with-db]]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [commandline.core :refer [print-help with-commandline]]
            [environ.core :refer [env]])
  (:gen-class))

(defn quotes-from-args
  "Return the quotes for the command line `args`."
  [db args]
  (if (empty? args)
    (quotes/quotes db)
    (quotes/by-symbols db args)))

(defn filename
  "Return the feature filename for `quote`."
  [quote format]
  (str (:id quote) "-" (:symbol quote) "." (name format)))

(defn path
  "Return the feature path for `quote`."
  [quote format & [opts]]
  (str (io/file (or (:directory opts) ".") (filename quote format))))

(defn output-format
  "Convert `s` to a keyword, representing the output format."
  [s]
  (some-> s name str/lower-case keyword))

(defn run
  "Export machine learning features into CSV files."
  [db & [{:keys [days directory format start end quotes] :as opts}]]
  (let [quotes (quotes-from-args db quotes)
        format (output-format (or format "edn"))]
    (assert (contains? #{:csv :edn :json} format) "Invalid output format.")
    (doseq [quote quotes :let [path (path quote format opts)]]
      (io/make-parents path)
      (let [result (features/save-features
                    db path {:days days
                             :end end
                             :format format
                             :quotes [quote]
                             :start start})]
        (log/infof "Saved %s features for %s to %s."
                   (:rows result) (:symbol quote) path)))))

(defn -main [& args]
  (with-commandline [quotes args]
    [[d days  "Select feature also DAYS backwards (default: 7)." :string "DAYS"]
     [e end "Select features until END time." :end "END"]
     [f format "Save features in FORMAT (CSV/EDN/JSON, default: EDN)."
      :string "FORMAT"]
     [h help "Print this help."]
     [r root "Save the in ROOT directory (default: features)." :string "ROOT"]
     [s start "Select features from START time." :time "START"]]
    (when help
      (print-help "antistock features [OPTION...]")
      (System/exit 0))
    (with-db [db (config/db env)]
      (run db {:days (or days 7)
               :directory (or root "features")
               :end end
               :format format
               :start start
               :quotes quotes}))))
