(ns antistock.accounts.twitter
  (:gen-class)
  (:require [antistock.accounts.anonbox :as anonbox]
            [antistock.accounts.system :refer [new-system]]
            [antistock.config.core :as config]
            [antistock.system :refer [with-component]]
            [antistock.util :refer [ask log-banner]]
            [clj-webdriver.taxi :as taxi]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [commandline.core :as cli]
            [crypto.random :as crypto-random]
            [datumbazo.core :as db]
            [datumbazo.db :refer [parse-url]]
            [environ.core :refer [env]]
            [faker.company :as company]
            [faker.internet :refer [domain-name]]
            [faker.name :refer [first-name last-name]]
            [inflections.core :refer [pluralize]]))

(defn try-fn [f webdriver elements & args]
  (let [elements (cond
                   (sequential? elements)
                   elements
                   (string? elements)
                   (taxi/find-elements webdriver {:css elements}))]
    (doseq [element elements]
      (try (apply f webdriver element args)
           (catch Exception _)))))

(defn parse-application-id
  "Parse the application id from `url`."
  [url]
  (when-let [[_ id] (re-matches #"https://apps.twitter.com/app/(\d+).*" (str url))]
    (bigint id)))

(def signup-url
  "The Twitter login url."
  "https://twitter.com/signup")

(defn full-name
  "Return a random full name."
  []
  (str (first-name)
       (when (rand-nth [true false])
         (str ", " (first-name)))
       ", " (last-name)))

(defn password
  "Return a random password."
  []
  (crypto-random/hex 16))

(defn input-text [webdriver query value]
  (doall (for [element (taxi/find-elements webdriver {:css query})]
           (try (taxi/clear webdriver element)
                (taxi/input-text webdriver element value)
                (catch Exception _)))))

(defn click [webdriver query]
  (doall (for [element (taxi/find-elements webdriver {:css query})]
           (try (taxi/click webdriver element)
                (catch Exception _)))))

(defn submit-login-form
  "Submit the login form."
  [webdriver account]
  (let [full-name (full-name), password (password)]
    (try-fn taxi/input-text webdriver "#full-name" full-name)
    (Thread/sleep 500)
    (try-fn taxi/input-text webdriver "#email" (:email account))
    (Thread/sleep 500)
    (try-fn taxi/input-text webdriver "#password" password)
    (Thread/sleep 500)
    (try-fn taxi/click webdriver "input.submit")
    (assoc account :full-name full-name :password password)))

(defn skip-telephone
  "Skip the telephone step."
  [webdriver]
  (click webdriver "a.skip-link"))

(defn skip-photo
  "Skip the telephone step."
  [webdriver]
  (taxi/find-elements webdriver {:css "a.js-skip-link"}))

(defn choose-username
  "Chosse a username."
  [webdriver account]
  (click webdriver "div.suggestions ul li button")
  (click webdriver "input.signup"))

(defn find-checkboxes [webdriver]
  (taxi/find-elements webdriver {:css "input[type=checkbox]"}))

(defn choose-interests
  "Chosse a username."
  [webdriver]
  (let [interests (shuffle (find-checkboxes webdriver))
        num-interests (rand-int (count interests))
        interests (take num-interests interests)]
    (doall (for [element interests]
             (try (taxi/click webdriver element)
                  (catch Exception _))))
    (click webdriver "button")))

(defn choose-followers
  "Chosse some random followers."
  [webdriver]
  (let [followers (shuffle (rest (find-checkboxes webdriver)))
        num-followers (rand-int (count followers))
        followers (take num-followers followers)]
    (try-fn taxi/click webdriver followers)
    (try-fn taxi/click webdriver "button")))

(defn parse-confirmation-link
  "Parse the account confirmation link from `s`."
  [s]
  (->> (str/split-lines (str s))
       (map #(re-matches #"(^\s*https://twitter.com/i/redirect[^\s]+)" %))
       (remove nil?)
       (first)
       (last)))

(defn confirm-account
  "Confirm the account by reading the confirmation link from the
  account's mailbox and following it."
  [webdriver account]
  (let [emails (anonbox/read-mailbox webdriver account)
        confirm-link (parse-confirmation-link emails)]
    (when-not confirm-link
      (throw (ex-info (str "Can't find Twitter confirmation link.")
                      {:account account})))
    (taxi/to webdriver confirm-link)
    account))

(defn create-account
  "Create a new Twitter account."
  [webdriver]
  (let [account (anonbox/create-account webdriver)]
    (taxi/to webdriver "https://www.google.com")
    (taxi/to webdriver signup-url)
    (let [account (submit-login-form webdriver account)]
      (skip-telephone webdriver)
      (choose-username webdriver account)
      (confirm-account webdriver account)
      account)))

(defn click-create-application-link
  "Click on the 'Create App' button."
  [webdriver]
  (->> (taxi/find-element webdriver {:xpath "//a[@href='/app/new']"})
       (taxi/click webdriver)))

(defn- application-name
  "Return a random application name."
  []
  (first (company/names)))

(defn- application-description
  "Return a random application description."
  []
  (company/catch-phrase))

(defn application-website
  "Return a random application website."
  []
  (str "https://" (domain-name)))

(defn fill-create-application-form
  "Fill in the create app form."
  [webdriver]
  (let [name (application-name)
        description (application-description)
        website (application-website)]
    (input-text webdriver "#edit-name" name)
    (input-text webdriver "#edit-description" description)
    (input-text webdriver "#edit-url" website)
    (when-not (taxi/selected? webdriver "#edit-tos-agreement")
      (taxi/toggle webdriver "#edit-tos-agreement"))
    {:name name
     :description description
     :website website}))

(defn find-application-id
  "Return the application id from the current url."
  [webdriver application]
  (some->> (taxi/current-url webdriver)
           (parse-application-id)
           (assoc application :id)))

(defn application-url
  "Return the url for the Twitter `application`."
  [application]
  (format "https://apps.twitter.com/app/%s" (:id application)))

(defn application-token-url
  "Return the url for the Twitter `application`."
  [application]
  (str (application-url application) "/keys"))

(defn application-settings-url
  "Return the url for the Twitter `application`."
  [application]
  (str (application-url application) "/settings"))

(defn application-access-token
  "Get the `application` settings."
  [webdriver application]
  (taxi/to webdriver (application-token-url application))
  (->> {:consumer-key (taxi/text webdriver ".app-settings .row:nth-child(1) span:nth-child(2)")
        :consumer-secret (taxi/text webdriver ".app-settings .row:nth-child(2) span:nth-child(2)")
        :access-token (taxi/text webdriver ".access .row:nth-child(1) span:nth-child(2)")
        :access-token-secret (taxi/text webdriver ".access .row:nth-child(2) span:nth-child(2)")}
       (merge application)))

(defn create-access-token
  "Create a Twitter application."
  [webdriver application]
  (log/infof "Creating access tokens for Twitter %s successfully created." (:name application))
  (taxi/to webdriver (format "https://apps.twitter.com/app/%s/keys"
                             (:id application)))
  (taxi/click webdriver (taxi/element webdriver "#edit-submit-owner-token"))
  (log/infof "Successfully created access tokens for Twitter application %s." (:name application))
  (application-access-token webdriver application))

(defn save-applications
  "Save `applications` to `db`."
  [db applications]
  (->> @(db/insert db :twitter.applications [:id :user-id :name :description]
          (db/values applications)
          (db/on-conflict [:id]
            (db/do-update
             {:name :EXCLUDED.name
              :description :EXCLUDED.description}))
          (db/returning :*))))

(defn applications
  "Return all applications of `account`."
  [webdriver account]
  (log/infof "Twitter applications of user %s:" (:screen-name account))
  (taxi/to webdriver "https://apps.twitter.com")
  (mapv (fn [id name description]
          (let [name (taxi/text name)
                description (taxi/text description)]
            (log/infof " - %s -%s" name description )
            {:id (parse-application-id (taxi/attribute id :href))
             :user-id (:id account)
             :name name
             :description description}))
        (taxi/find-elements webdriver {:css "ul.apps-list li .app-details h2 a"})
        (taxi/find-elements webdriver {:css "ul.apps-list li .app-details h2"})
        (taxi/find-elements webdriver {:css "ul.apps-list li .app-details p"})))

(defn application-by-name
  "Find the Twitter application by `name`."
  [webdriver account name]
  (->> (applications webdriver account)
       (filter #(= (:name %) name))
       (first)))

(defn create-application
  "Create a Twitter application."
  [webdriver account]
  (log/info "Creating new Twitter application ...")
  (taxi/to webdriver "https://apps.twitter.com")
  (click-create-application-link webdriver)
  (let [application (fill-create-application-form webdriver)]
    (taxi/click webdriver "#edit-submit")
    (Thread/sleep 2000)
    (log/infof "Twitter application %s successfully created." (:name application))
    (->> (application-by-name webdriver account (:name application))
         (create-access-token webdriver))))

(defn phone-number
  "Return a random phone number."
  []
  (str (rand-nth ["0155" "0157" "0163" "0177" "0178"
                  "0151" "0160"  "0170" "0171" "0175"])
       (apply str (repeatedly 7 #(inc (rand-int 8))))))

(defn save-account
  "Save the Twitter account."
  [db account]
  (->> @(db/insert db :twitter.users [:id :name :email :screen-name]
          (db/values [account])
          (db/on-conflict [:email]
            (db/do-update
             {:name :EXCLUDED.name
              :screen-name :EXCLUDED.screen-name}))
          (db/returning :*))
       (first)
       (merge account)))

(defn settings
  "Go to settings and add them to `account`."
  [webdriver account]
  (taxi/to webdriver "https://twitter.com/settings/account")
  (->> {:email (taxi/value webdriver "#user_email")
        :name (taxi/text webdriver ".DashboardProfileCard-name")
        :screen-name (taxi/value webdriver "#user_screen_name")}
       (merge account)))

(defn find-twitter-id
  "Find the Twitter user id of the current user and add it to `account`."
  [webdriver account]
  (log/infof "Looking up the Twitter user id of %s ..." (:screen-name account))
  (taxi/to webdriver "http://mytwitterid.com")
  (taxi/input-text webdriver "input[name=screen-name]" (:screen-name account))
  (taxi/click webdriver "input[type=submit]")
  (let [selector ".result-id > strong"]
    (doseq [n (range 10) :when (not (taxi/element webdriver selector))]
      (Thread/sleep 1000))
    (if-let [id (re-matches #"\d+" (taxi/text webdriver selector))]
      (assoc account :id (bigint id)))))

(defn login
  "Login to Twitter using `account`."
  [webdriver account]
  (log/infof "Logging into Twitter as %s ..." (:email account))
  (taxi/to webdriver "https://twitter.com/login")
  (input-text webdriver "input.email-input" (:email account))
  (input-text webdriver "input[type=password]" (:password account))
  (taxi/click webdriver "button[type=submit]")
  (if (re-matches #"https://twitter\.com/login/error.*"
                  (taxi/current-url webdriver))
    (throw (ex-info "Can't log into Twitter. Invalid credentials" account))
    (do (log/info "Successfully logged into Twitter. Loading user details ...")
        (->> (settings webdriver account)
             (find-twitter-id webdriver)))))

(defn logout
  "Logout of Twitter."
  [webdriver]
  (taxi/to webdriver "https://twitter.com")
  (when-let [element (taxi/element webdriver "#user-dropdown-toggle")]
    (taxi/click element)
    (taxi/click webdriver "#signout-button > button")))

(defn download-url
  "Download `uri` to `filename`."
  [uri file]
  (with-open [in (io/input-stream uri)
              out (io/output-stream file)]
    (io/copy in out)))

(defn random-giphy
  "Return a random GIF from Giphy."
  [webdriver]
  (taxi/to webdriver "http://giphy.com/search/random")
  (let [figures (taxi/find-elements webdriver {:css "figure"})
        figure (taxi/click webdriver (rand-nth figures))
        element (taxi/element webdriver ".gif-figure img")]
    (taxi/attribute element :src)))

(defn upload-application-picture
  "Upload `filename` as picture for the Twitter `application`."
  [webdriver application filename]
  (let [filename (.getAbsolutePath (io/file filename))]
    (taxi/to webdriver (application-settings-url application))
    (let [element (taxi/element webdriver "#edit-image")]
      (taxi/send-keys element filename)
      (taxi/click webdriver "#edit-submit"))))

(defn synchronize-account
  "Synchronize the Twitter `account`."
  [system account]
  (log/infof "Synchronizing Twitter account of user %s ..." (:email account))
  (let [account (login (:webdriver system) account)]
    (db/with-connection [db (:db system)]
      (let [account (save-account db account)]
        (log/infof "Successfully synchronized Twitter account of user %s." (:email account))
        account))))

(defn synchronize-applications
  "Synchronize the applications of the Twitter `account`."
  [system account]
  (log/infof "Synchronizing Twitter applications of user %s ..." (:email account))
  (let [applications (applications (:webdriver system) account)]
    (db/with-connection [db (:db system)]
      (let [applications (save-applications db applications)]
        (log/infof "Successfully synchronized %s of user %s."
                   (pluralize (count applications) "Twitter application")
                   (:screen-name account))
        applications))))

(defn config
  "Return the config for `options`."
  [opts]
  (let [browser (keyword (or (:browser opts) :chrome))]
    {:db (or (some-> opts :db parse-url) (config/db env))
     :webdriver {:browser browser}}))

(defn run-cli
  "Synchronize a Twitter account and all it's applications."
  [{:keys [webdriver] :as system} account & [opts]]
  (logout webdriver)
  (let [account (synchronize-account system account)]
    (when (:create-app opts)
      (create-application webdriver account))
    (synchronize-applications system account)))

(defn -main [& args]
  (log-banner)
  (cli/with-commandline [[options [username password]] args]
    [[b browser "The web browser to be used, either `chrome`, or `firefox`." :string "BROWSER"]
     [c create-app "Create a new random Twitter applications."]
     [d db "The database URL." :string "DATABASE-URL"]
     [h help "Print this help."]]
    (if (:help options)
      (cli/print-help "twitter-account USERNAME PASSWORD")
      (with-component [system (new-system (config options))]
        (run-cli system
                 {:email (ask "Please enter your Twitter username: " username)
                  :password (ask "Please enter your Twitter password: " password)}
                 options)))))

(comment

  (def webdriver (:webdriver reloaded.repl/system))

  (logout webdriver)

  (def my-account
    (->> {:email "b4pfrhrobp"
          :password "b4pfrhrobp13+"}
         (login webdriver)))

  (def my-apps
    (applications webdriver my-account))

  (run-cli reloaded.repl/system my-account)

  (download-url "https://media.giphy.com/media/3o6Mbdfqw4FDh1mGk0/giphy.gif" "X.gif")
  (prn (random-giphy webdriver))
  (prn (upload-application-picture webdriver (first my-apps) "giphy.gif")))
