(ns dateutils.core
  (:require [cljc.java-time.local-date :as ld]
            [clojure.spec.alpha :as s]))

;; all the possible formats supported
(s/def ::ms-since-epoch (s/nilable pos-int?))
(s/def ::seconds-since-epoch (s/nilable pos-int?))
(s/def ::date-str (s/nilable string?))
(s/def ::datetime-str (s/nilable string?))
(s/def ::days-since-epoch (s/nilable pos-int?))

(def ms-in-day (* 24 60 60 1000))

;; this is the place where all the transformations are defined
;; All the other transformations are automatically generated
;; by composing these simpler transformations.
(def conv
  {[::seconds-since-epoch ::ms-since-epoch] [#(* 1000 %)
                                             #(/ % 1000)]

   [::ms-since-epoch ::days-since-epoch]    [#(/ % ms-in-day)
                                             #(* % ms-in-day)]

   [::days-since-epoch ::date-str]          [#(str (ld/of-epoch-day %))
                                             #(ld/to-epoch-day (ld/parse %))]})

(def forward
  (into {}
        (for [[t [f1 _]] conv]
          [t f1])))

(def backward
  (into {}
        (reverse
         (for [[t [_ f2]] conv]
           [(reverse t) f2]))))

(def conversions-flattened
  (merge backward forward))

(def all-formats (-> conv
                     keys
                     flatten
                     distinct))

(defn flow
  [c]
  (-> c
      keys
      flatten
      distinct))

(defn- take-until
  "Similar to take-while but keep the last element as well"
  [coll v]
  ;; when it's not there is just nil
  (when (contains? (set coll) v)
    (conj (vec (take-while (complement #{v}) coll)) v)))

(defn get-subs
  [from-type to-type c]
  ;; all the intermediate subs also need to exist
  (let [trimmed-left (drop-while (complement #{from-type}) (flow c))]
    (partition 2 1 (take-until trimmed-left to-type))))

(defn get-conversion
  [c]
  (if-not (contains? conversions-flattened c)
    (throw (Exception. (str "Could not find conversion" (seq c))))
    (let [conv-fn (get conversions-flattened c)]
      ;; deal with possible nil values here
      #(some-> % conv-fn))))

(defn- get-transformation*
  [from-type to-type]
  (if (= from-type to-type)
    identity
    (let [subs-f (get-subs from-type to-type forward)
          subs-b (get-subs from-type to-type backward)
          subs (last (sort-by count [subs-f subs-b]))]
      (apply comp (map get-conversion subs)))))

;; to make sure it's only called once
(def get-transformation (memoize get-transformation*))

(defn- valid-type?
  [typ]
  (or (contains? (set all-formats) typ)
      (contains? (set all-formats) typ)))

(defn convert
  [from-type to-type val]
  {:pre [(and (valid-type? from-type) (valid-type? to-type))]}
  (let [f (get-transformation from-type to-type)]
    ;; make sure the input type is correct as well, this has only
    ;; effect if the global `check-asserts` is true
    (s/assert from-type val)
    (f val)))

(s/fdef convert
  :args (s/cat :from-type keyword? :to-type keyword? :val any?)
  :ret any?)
