(ns alumbra.canonical.selection-set
  (:require [alumbra.canonical.value :refer [resolve-value]]))

(declare resolve-selection-set)

;; ## Helpers

(defn- field-key
  [{:keys [alumbra/field-alias
           alumbra/field-name]}]
  (or field-alias field-name))

(defn- add-type-condition
  [{:keys [type-condition]} field]
  (if type-condition
    (assoc field :type-condition type-condition)
    field))

(defn- field-type-of
  [{:keys [schema scope-type]} {:keys [alumbra/field-name]}]
  (let [kind (get-in schema [:type->kind scope-type])
        type (case kind
               :type      (get-in schema [:types scope-type])
               :interface (get-in schema [:interfaces scope-type])
               :union     (get-in schema [:unions scope-type]))]
    (get-in type [:fields field-name])))

(defn- generate-nested-selection
  [{:keys [type-description
           type-name
           non-null?]}
   selection]
  (if type-name
    {:field-type    :object
     :non-null?     non-null?
     :selection-set selection}
    {:field-type :list
     :non-null?  non-null?
     :field      (generate-nested-selection type-description selection)}))

(defn- generate-nested-leaf
  [{:keys [type-name
           type-description
           non-null?]}]
  (if type-description
    {:field-type :list
     :non-null?  non-null?
     :field      (generate-nested-leaf type-description)}
    {:field-type :leaf
     :non-null?  non-null?}))

;; ## Field Resolution

(defn- leaf?
  [{:keys [alumbra/selection-set]}]
  (not selection-set))

(defn- data-for-leaf
  [_ {:keys [type-description]} _]
  (generate-nested-leaf type-description))

(defn- data-for-arguments
  [opts _ {:keys [alumbra/arguments]}]
  (->> (for [{:keys [alumbra/argument-name
                     alumbra/argument-value]} arguments]
         [argument-name (resolve-value opts argument-value)])
       (into {})
       (hash-map :arguments)))

(defn- data-for-subselection
  [opts
   {:keys [type-description
           type-name]}
   {:keys [alumbra/selection-set]}]
  (->> (resolve-selection-set
         (assoc opts
                :scope-type     type-name
                :type-condition nil)
         selection-set)
       (generate-nested-selection type-description)))

(defn- resolve-field*
  [opts field]
  (let [field-type (field-type-of opts field)]
    ;; TODO: attach expected value type
    (merge
      {:field-name (:alumbra/field-name field)}
      (data-for-arguments opts field-type field)
      (if (leaf? field)
        (data-for-leaf opts field-type field)
        (data-for-subselection opts field-type field)))))

(defn- resolve-field
  [result opts field]
  (->> (resolve-field* opts field)
       (add-type-condition opts)
       (assoc result (field-key field))))

;; ## Inline Spread Resolution
;;
;; Inline spreads are merged into the current selection set, adding a type
;; condition to each field.

(defn- resolve-inline-spread
  [result opts {:keys [alumbra/type-condition alumbra/selection-set]}]
  (let [fragment-type-name (:alumbra/type-name type-condition)]
    (->> (resolve-selection-set
           (assoc opts
                  :scope-type     fragment-type-name
                  :type-condition fragment-type-name)
           selection-set)
         (merge result))))

;; ## Named Spread Resolution
;;
;; Named spreads are inlined directly using the preprocessed fragment selection
;; sets.

(defn- resolve-named-spread
  [result {:keys [fragments]} {:keys [alumbra/fragment-name]}]
  (merge result (get fragments fragment-name)))

;; ## Selection Set Traversal

(defn resolve-selection-set
  [opts selection-set]
  (reduce
    (fn [result selection]
      (condp #(contains? %2 %1) selection
        :alumbra/fragment-name  (resolve-named-spread result opts selection)
        :alumbra/type-condition (resolve-inline-spread result opts selection)
        :alumbra/field-name     (resolve-field result opts selection)))
    {} selection-set))
