(ns allgress.cereus.core
  (:require-macros [freactive.macros :refer [rx non-reactively]]
                   [cljs.core.async.macros :refer [go]])
  (:require [clojure.set]
            [cljs.core.async :refer [<! chan mult tap]]
            [freactive.dom :as dom]))

(enable-console-print!)

;;; TODO - when laziness bug is fixed in freactive, remove this
(defn add-watch
  [iref key f]
  #_@iref
  #_@iref
  (cljs.core/add-watch iref key f))

(defn- add-property-binding
  [element prop-name attr-val attribute-changed]
  (let [new-val @attr-val
        bindings (if-let [b (aget element "cereus-bindings")]
                   b
                   (let [b (atom {})]
                     (aset element "cereus-bindings" b)
                     b))]
    (when-not (contains? @bindings prop-name)
      (aset element prop-name new-val)
      (.call attribute-changed element prop-name nil new-val)
      (add-watch attr-val prop-name
                 (fn [k _ old-val new-val]
                   (aset element prop-name attr-val)
                   (.call attribute-changed element prop-name old-val new-val)))
      (swap! bindings assoc prop-name attr-val))))

(defn register-property-namespace
  ([namespace]
   (dom/register-attr-prefix!
     namespace
     (fn [element attr-name attr-val]
       (let [prop-name (str namespace "/" attr-name)
             old-val (aget element prop-name)
             attribute-changed (aget element "attributeChangedCallback")]
         (if (= (type attr-val) freactive.core/ReactiveExpression)
           (add-property-binding element prop-name attr-val attribute-changed)
           (do
             (aset element prop-name attr-val)
             (.call attribute-changed element prop-name old-val attr-val))))))))

(let [element-prototypes (aget js/window "element-prototypes")]
  (when (nil? element-prototypes)
    (aset js/window "element-prototypes" (atom {}))))

(let [waiting-for-registration (aget js/window "waiting-for-registration")]
  (when (nil? waiting-for-registration)
    (aset js/window "waiting-for-registration" (atom #{}))))

(defonce notification-channels (atom {}))

(defonce reload-listeners (atom #{}))

(defn waiting-for
  []
  (let [waiting-for-registration (aget js/window "waiting-for-registration")]
    (println @waiting-for-registration)))

(defn on-jsload
  []
  (doseq [f @reload-listeners] (f)))

(defn- keyword->str*
  [x]
  (str (if (namespace x) (str (namespace x) "/")) (name x)))

(def keyword->str (memoize keyword->str*))

(defn- map->js
  [x]
  (apply js-obj (mapcat (fn [[k v]] [(keyword->str k) v]) x)))

(defn fire!
  ([el event-type]
   (fire! el event-type {:bubbles true :cancelable true}))
  ([el event-type event-init]
   (let [type (cond
                (keyword? event-type) (name event-type)
                (string? event-type) event-type)
         event-data (js/CustomEvent. type (map->js event-init))]
     (.dispatchEvent el event-data))))

(defn listen!
  [el event-type listener]
  (let [type (cond
               (keyword? event-type) (name event-type)
               (string? event-type) event-type)]
    (.addEventListener el type listener)))

(defn unlisten!
  [el event-type listener]
  (let [type (cond
               (keyword? event-type) (name event-type)
               (string? event-type) event-type)]
    (.removeEventListener el type listener)))

(defn nodelist-to-seq
  "Converts nodelist to (not lazy) seq."
  [nl]
  (let [result-seq (if (array? nl)
                     (map (fn [i] (aget nl i)) (range (.-length nl)))
                     (map #(.item nl %) (range (.-length nl))))]
    (doall result-seq)))

;;; spec fields
;;; :state - REQUIRED (fn []) which returns an atom containing initial state
;;; :view - REQUIRED (fn [state this]) to render view
;;; :created - (fn [state this]) called after element is first constructed
;;; :attached - (fn [state this]) called when element is added to DOM
;;; :after-render (fn [state this]) called after view is initially rendered
;;; :detached - (fn [state this]) called when element is removed from DOM
;;; :attribute-changed (fn [state this attr-name old-val new-val]) called when attribute changes
;;; :root - (fn [state this]) DOM root to mount view (e.g. (fn [state this] (.createShadowRoot this))
(defn- spec-to-proto
  ([base-element spec]
   (assert (and (some? (:state spec)) (some? (:view spec))) "Custom element spec must contain :state and :view functions")
   (let [scope-name (str (gensym "cereus/scope"))]
     (.create
       js/Object (.-prototype base-element)
       (clj->js
         {:spec spec
          :createdCallback
                {:value
                 (fn []
                   (this-as this
                     (let [reload-listener (atom nil)
                           scope (merge {:this            this
                                         :state           ((:state spec))
                                         :reload-listener reload-listener
                                         :listen-reload   (fn [f]
                                                            (cljs.core/swap! reload-listeners cljs.core/conj f)
                                                            (cljs.core/reset! reload-listener f))}
                                        (if-let [s (:scope spec)]
                                          (if (fn? s) (s) s)
                                          {}))]
                       (aset this scope-name scope)
                       (when-let [created-fn# (:created spec)]
                         (non-reactively (created-fn# (:state scope) this)))
                       (when-let [attribute-changed-fn (:attribute-changed spec)]
                         (doseq [attr (nodelist-to-seq (aget this "attributes"))]
                           (attribute-changed-fn (:state scope) this (aget attr "name") nil (aget attr "value")))))))}
          :attachedCallback
                {:value
                 (fn []
                   (this-as this
                     (let [scope (aget this scope-name)
                           state (:state scope)
                           view (:view spec)
                           root (if (:root spec)
                                  ((:root spec) state this)
                                  (let [root (.createElement js/document "div")]
                                    (.appendChild this root)
                                    root))]
                       (let [shadow-root (.-shadowRoot this)]
                         (if (some? shadow-root)
                           (set! (.-cereus-root this) shadow-root)
                           (set! (.-cereus-root this) root)))
                       (when-let [attached-fn (:attached spec)]
                         (non-reactively (attached-fn state this)))
                       (non-reactively
                         (dom/mount! root (view state this)))
                       ((:listen-reload scope) (fn []
                                                  (when (:render-on-reload? spec)
                                                    (dom/mount! root ((:view spec) state this)))))
                       (when-let [after-render-fn (:after-render spec)]
                         (after-render-fn state this)))))}
          :detachedCallback
                {:value
                 (fn []
                   (this-as this
                     (let [scope (aget this scope-name)]
                       (when-let [detached-fn (:detached spec)]
                         (non-reactively (detached-fn (:state scope) this)))
                       (dom/unmount! (.-cereus-root this))
                       (when-let [f (:reload-listener scope)]
                         (swap! reload-listeners disj f)))))}
          :attributeChangedCallback
                {:value
                 (fn [attr-name old-value new-value]
                   (this-as this
                     (let [scope (aget this scope-name)]
                       (when-let [attribute-changed-fn (:attribute-changed spec)]
                         (non-reactively (attribute-changed-fn (:state scope) this attr-name old-value new-value))))))}})))))

(defn- finalize-registration
  [tag-name base-element spec]
  (let [element-prototypes (aget js/window "element-prototypes")
        waiting-for-registration (aget js/window "waiting-for-registration")
        js-prototype (spec-to-proto base-element spec)]
    (register-property-namespace (name tag-name))
    (.registerElement js/document (name tag-name) (clj->js {:prototype js-prototype}))
    (swap! waiting-for-registration disj tag-name)
    (swap! element-prototypes assoc tag-name js-prototype)
    (swap! notification-channels dissoc tag-name)
    tag-name))

(defn- update-registration
  [tag-name base-element spec]
  (let [element-prototypes (aget js/window "element-prototypes")
        registered-prototype (@element-prototypes tag-name)
        registered-spec (aget registered-prototype "spec")]
    (when (not (identical? registered-spec spec))
      (let [js-prototype (spec-to-proto base-element spec)]
        (aset registered-prototype "spec" spec)
        (aset registered-prototype "createdCallback" (aget js-prototype "createdCallback"))
        (aset registered-prototype "attachedCallback" (aget js-prototype "attachedCallback"))
        (aset registered-prototype "detachedCallback" (aget js-prototype "detachedCallback"))
        (aset registered-prototype "attributeChangedCallback" (aget js-prototype "attributeChangedCallback")))))
  tag-name)

(defn- register-element
  [tag-name base-element spec register-deps-fn]
  (let [element-prototypes (aget js/window "element-prototypes")
        waiting-for-registration (aget js/window "waiting-for-registration")]
    (if (some? (@element-prototypes tag-name))
      (go
        (update-registration tag-name base-element spec))   ; TODO - also pass register-deps-fn?
      (if (some? (@waiting-for-registration tag-name))
        (let [ch (chan)]
          (tap (@notification-channels tag-name) ch true)
          ch)
        (let [c (go
                  (<! (register-deps-fn))
                  (finalize-registration tag-name base-element spec))
              m (mult c)
              ch (chan)]
          (swap! waiting-for-registration conj tag-name)
          (swap! notification-channels assoc tag-name m)
          (tap m ch true)
          ch)))))

(defn register-custom-element!
  ([tag-name base-element spec]
   (register-custom-element! tag-name base-element spec (fn [] (go tag-name))))
  ([tag-name base-element spec register-deps-fn]
   (register-element tag-name base-element spec register-deps-fn)))

