(set! *warn-on-reflection* true)

(ns alandipert.enduro
  (:refer-clojure :exclude [atom swap! reset!])
  (:require [clojure.java.io :as io]
            [clojure.core :as core])
  (:import (java.util.concurrent.atomic AtomicReference)
           (java.io File FileOutputStream OutputStreamWriter PushbackReader)
           (java.nio.channels Channels)))

;;; Durable resources

(defprotocol IDurable
  "Represents a durable resource."
  (exists? [this] "Whether or not the resource exists already.")
  (contents [this] "Read Clojure data from the resource, returning nil if empty.")
  (commit! [this value] "Attempt to commit value to resource, returning true if successful."))

(defn read-file
  "Reads the first Clojure expression from f, returning nil if f is empty."
  [^File f]
  (with-open [pbr (PushbackReader. (io/reader f))]
    (read pbr false nil)))

(deftype FileBackend [^File f]
  IDurable
  (exists? [this] (.exists f))
  (contents [this] (read-file f))
  (commit!
    [this value]
    (with-open [fos (FileOutputStream. f)]
      (let [channel (.getChannel fos)]
        (when (.tryLock channel)
          (with-open [os (OutputStreamWriter. (Channels/newOutputStream channel))]
            (print-method value os)
            (.flush os)
            true))))))

;;; EnduroAtom

(defprotocol IAtom
  (-swap! [a f args])
  (-reset! [a newval]))

(defn validate
  [f v]
  (if f (or (f v)
            (throw (IllegalStateException. "Invalid reference state")))))

(defn notify-watches
  [a watches v newv]
  (doseq [[k f] watches]
    (try
      (f k a v newv)
      (catch Exception e
        (throw (RuntimeException. e))))))

(deftype EnduroAtom [meta
                     watches
                     validator
                     ^alandipert.enduro.IDurable resource
                     ^AtomicReference state]
  clojure.lang.IMeta
  (meta [a] meta)
  clojure.lang.IRef
  (setValidator [a vf] (core/reset! validator vf))
  (getValidator [a] @validator)
  (getWatches [a] @watches)
  (addWatch [a k f] (do (core/swap! watches assoc k f) a))
  (removeWatch [a k] (do (core/swap! watches dissoc k) a))
  clojure.lang.IDeref
  (deref [a] (.get state))
  IAtom
  (-swap! [a f args]
    (loop []
      (let [v (deref a)
            newv (apply f v args)]
        (validate @validator newv)
        (if (and (.compareAndSet state v newv)
                 (commit! resource newv))
          (do (notify-watches a @watches v newv) newv)
          (recur)))))
  (-reset! [a v]
    (-swap! a (constantly v) ())))

(defn swap!
  "Atomically swaps the value of enduro-atom to be:
   (apply f current-value-of-atom args) and commits the value to disk.

  Note that f may be called multiple times, and thus should be free of
  side effects.  Returns the value that was swapped in."
  [enduro-atom f & args]
  (-swap! enduro-atom f args))

(defn reset!
  "Sets the value of durable-atom to newval without regard for the
  current value.  Returns newval."
  [enduro-atom newval]
  (-reset! enduro-atom newval))

(defn atom
  "Creates and returns a durable atom.  If file exists and is not
  empty, it is read and becomes the initial value.  Otherwise, the
  initial value is init and a new file is created and written to.

  Path may be any resource understood by clojure.java.io/file.

  Takes zero or more additional options:

  :meta metadata-map

  :validator validate-fn

  If metadata-map is supplied, it will be come the metadata on the
  atom. validate-fn must be nil or a side-effect-free fn of one
  argument, which will be passed the intended new state on any state
  change. If the new state is unacceptable, the validate-fn should
  return false or throw an exception."
  [init file & {:keys [meta validator]
                :or {meta {} validator nil}}]
  (let [res (FileBackend. (io/file file))]
    (doto (EnduroAtom. meta
                       (core/atom {})
                       (core/atom validator)
                       res
                       (AtomicReference. (if (exists? res)
                                           (or (contents res) init)
                                           init)))
      (swap! identity))))