(ns boot-figwheel.config
  "
https://github.com/bhauman/lein-figwheel/blob/master/sidecar/resources/conf-fig-docs/FigwheelOptions.txt
"
  (:require
   [clojure.java.io :as jio]
   [boot.file]
   )
  (:import
   clojure.lang.MapEntry
   ))

(def ^:dynamic *js-suffix* ".js")
(def ^:dynamic *outdir-suffix* ".out")

(defn- opt-none?
  "Given a map of compiler options returns true if a build will be
  compiled in :optimizations :none mode"
  [{:keys [optimizations]}]
  (or (nil? optimizations) (= optimizations :none)))

(defmulti update-build-options
  (fn
    ([build ks] ks)
    ([build ks x] ks)
    ([build ks x y] ks)
    ([build ks x y & args] ks)))

(defmethod update-build-options [:compiler :output-to]
  [{:keys [id] :as build} ks target-path]
  (assert (some? id))
  (-> build
    (update-in ks
      (fn [output-to]
        (as-> (if (string? output-to) output-to (str id *js-suffix*))
          output-to
          (.getPath (jio/file target-path output-to)))))))

(defmethod update-build-options [:compiler :output-dir]
  [{:keys [id]
    {:keys [output-to                   ; dependent on [:compiler :output-to]
            output-dir]
     :as compiler} :compiler
    :as build}
   ks
   target-path]
  (let [parent     (boot.file/parent output-to)
        output-dir (cond
                     (string? output-dir) (jio/file parent output-dir)
                     (opt-none? compiler) (jio/file parent (str id *outdir-suffix*))
                     :else                output-dir)]
    (-> build
      (update-in ks
        #(if (instance? java.io.File output-dir)
           (.getPath output-dir)
           %)))))

(defmethod update-build-options [:compiler :asset-path]
  [{:keys[id]
    {:keys [output-dir                  ; dependent on [:compiler :output-dir]
            asset-path]
     :as compiler} :compiler
    :as build}
   ks
   target-path]
  (let [asset-path (cond
                     (string? asset-path) (jio/file asset-path)
                     (opt-none? compiler) (boot.file/relative-to target-path output-dir)
                     :else                asset-path)]
    (-> build
      (update-in ks
        #(if (instance? java.io.File asset-path)
           (.getPath asset-path)
           %)))))

(defmethod update-build-options [:compiler :source-map]
  [build ks target-path]
  (let [source-map (get-in build ks)]
    (if (string? source-map)
      (let [output-to (get-in build [:compiler :output-to]) ; dependent on [:compiler :output-to]
            parent    (boot.file/parent output-to)]
        (-> build
          (assoc-in ks
            (.getPath (jio/file parent source-map)))
          #_(assoc-in [:compiler :source-map-path]
              (.getPath (boot.file/relative-to target-path parent)))))
      build)))

(defn update-fw-config
  [config target-path]
  (update config :builds
    (fn [builds]
      (if (map? builds)
        (into {}
          (map
           (fn [[id build]]
             (MapEntry.
              id
              (-> build
                (assoc :id (name id))
                ;; order is important
                (update-build-options [:compiler :output-to] target-path)
                (update-build-options [:compiler :source-map] target-path)
                (update-build-options [:compiler :output-dir] target-path)
                (update-build-options [:compiler :asset-path] target-path)
                (dissoc :id))))
           builds))
        (mapv
         (fn [build]
           (-> build
             ;; order is important
             (update-build-options [:compiler :output-to] target-path)
             (update-build-options [:compiler :source-map] target-path)
             (update-build-options [:compiler :output-dir] target-path)
             (update-build-options [:compiler :asset-path] target-path)))
         builds)))))

(comment
  (= (try (update-build-options {} [:compiler :output-to] "target")
          (catch Throwable e (class e)))
     java.lang.AssertionError)
  (update-build-options {:id "dev"} [:compiler :output-to] "target")

  (update-build-options {:id "dev"} [:compiler :source-map] "target") {:id "dev"}
  (-> {:id "dev"}
    (update-build-options [:compiler :output-to] "target")
    (update-build-options [:compiler :source-map] "target"))

  (-> {:id "dev" :compiler {:source-map "a.map"}}
    (update-build-options [:compiler :output-to] "target")
    (update-build-options [:compiler :source-map] "target"))
  )
