(ns ags799.boot-docker
  {:boot/export-tasks true}
  (:require [boot.core :as boot]
            [clojure.java.shell :refer [sh]]
            [clojure.java.io :as io]))

(defn- default-dockerfile [jar-path]
  (let [entrypoint (format "ENTRYPOINT [\"java\",\"-jar\",\"%s\"]" jar-path)]
    (clojure.string/join "\n" ["FROM openjdk:8-jre"
                               (format "COPY %s ." jar-path)
                               entrypoint])))

(defn- spit-default-dockerfile [dockerfile jar-path]
  (spit dockerfile (default-dockerfile jar-path)))

(defn- find-jar [files]
  (boot/tmp-path (first (boot/by-ext [".jar"] files))))

(boot/deftask dockerfile
  "Creates a generic Dockerfile at the root of the project.

  Note that your application may require a custom Dockerfile, in which case
  you shouldn't use this step at all.

  The generated Dockerfile loads a jar onto a Java 8 JRE image and sets the
  entrypoint to `java -jar the-jar.jar`."
  []
  (boot/with-pre-wrap fileset
    (let [jar-path (find-jar (boot/output-files fileset))
          tmp (boot/tmp-dir!)
          dockerfile (io/file tmp "Dockerfile")]
      (spit-default-dockerfile dockerfile jar-path)
      (boot/commit! (boot/add-resource fileset tmp)))))

(boot/deftask docker-image
  "Builds a Docker image from the target/ path.

  Note that the name-str parameter is required.

  The target/ path is used in order to pair well with boot's built-in target
  task. Docker doesn't seem to play nicely with symlinks, so we ask that you
  write your build artifacts to target/ before trying to build a Docker image."
  [n name-str VAL str "name of the Docker image"]
  (let [output (sh "docker" "build" "-t" name-str "target")]
    (if (zero? (:exit output))
      identity
      (throw (Exception. (str "non-zero exit code: " output))))))
