(ns adzerk.clj-kinesis-producer.worker
  (:require
    [adzerk.clj-kinesis-producer.queue :as queue]))

(defmacro with-callbacks
  [success error & body]
  `(try (~success (do ~@body))
        (catch Throwable t#
          (try (~error t#) (catch Throwable _#)))))

(defn num-inflight!
  [atom]
  (count (swap! atom (partial remove realized?))))

(defprotocol IWorker
  (start! [this])
  (stop!  [this])
  (stats  [this])
  (put!   [this data]))

(defrecord Worker
  [queue topic process-job on-success on-error take-timeout put-timeout max-in-flight in-flight running?]
  IWorker
  (start! [this]
    (let [sym (reset! running? (gensym))]
      (future
        (while (= @running? sym)
          (with-callbacks identity on-error
            (if (< max-in-flight (num-inflight! in-flight))
              (Thread/sleep 100)
              (swap! in-flight conj
                     (future (with-callbacks on-success on-error
                               (queue/take! queue topic take-timeout process-job))))))))))
  (stop! [this]
    (reset! running? false))
  (stats [this]
    (let [qstat (get (queue/stats queue) (name topic))]
      (merge {:in-flight (num-inflight! in-flight)} qstat)))
  (put! [this data]
    (with-callbacks identity on-error
      (queue/put! queue topic data put-timeout))))

(defn worker
  [queue topic process-job on-success on-error take-timeout put-timeout max-in-flight]
  (Worker. queue topic process-job on-success on-error take-timeout put-timeout max-in-flight (atom ()) (atom false)))
