(ns coendou.semaphore
  (:require
   [coendou.protocols :as prot]
   [manifold.stream :as s]))

(defrecord Semaphore [stream limit-state]
  prot/ISemaphore
  (acquire! [_]
    (s/take! stream))

  (release! [_]
    (s/put! stream (Object.))))

(defn limit [s]
  @(:limit-state s))

(defn in-flight [s]
  (- (limit s) (:buffer-size (s/description (:stream s)))))

(defn- set-limit!* [limit-state new-limit]
  (assert (>= new-limit 0) (str "Limit should be greater than or equal to zero"))
  (let [old-limit @limit-state]
    (if (compare-and-set! limit-state old-limit new-limit)
      (- new-limit old-limit)
      (throw (ex-info "Concurrent limit update detected" {})))))

(defn- add-in-flight! [stream n]
  (if (pos? n)
    (s/put-all! stream (repeat n (Object.)))
    (dotimes [i (- n)]
      (s/take! stream))))

(defn set-limit! [x value]
  (let [delta (set-limit!* (:limit-state x) value)]
    (add-in-flight! (:stream x) delta)))

(defn acquire! [s]
  (prot/acquire! s))

(defn release! [s]
  (prot/release! s))

(defn semaphore? [x]
  (instance? Semaphore x))

;; TODO add spec for limit
(defn create [initial-limit]
  (let [s (s/stream Integer/MAX_VALUE)]
    (->Semaphore (doto s
                   (s/put-all! (repeat initial-limit (Object.))))
                 (atom initial-limit))))
