(ns coendou.reporting
  (:require
   [coendou.tdigest :as tdigest]
   [coendou.specs]
   [clojure.spec :as spec]))

(require '[clojure.spec.test :as stest])
;; SPECS

(defn xf-partition-with-flush
  "Returns a lazy sequence of lists like partition, but may include
  partitions with fewer than n items at the end.  Returns a stateful
  transducer when no collection is provided."
  {:added "1.2"
   :static true}
  ([^long n sentinel]
   (fn [rf]
     (let [a (java.util.ArrayList. n)]
       (fn
         ([] (rf))
         ([result]
            (let [result (if (.isEmpty a)
                           result
                           (let [v (vec (.toArray a))]
                             ;;clear first!
                             (.clear a)
                             (unreduced (rf result v))))]
              (rf result)))
         ([result input]
          (if (= input sentinel)
            (let [v (vec (.toArray a))]
                  (.clear a)
                  (rf result v))
            (do
              (.add a input)
              (if (= n (.size a))
                (let [v (vec (.toArray a))]
                  (.clear a)
                  (rf result v))
                result)))))))))

(defn update-counts [counts event]
  (update counts event (fnil inc 0)))

(defn stats [vs]
  (let [[{:keys [prefix id]}] vs
        counts (reduce update-counts {} (map :type vs))
        latencies (map :elapsed vs)]
    ;; ADD assert that all cmd's and prefixes should be the same?
    {:prefix prefix
     :cmd id
     :count (count vs)
     :counts counts
     :sum-latency-in-nano (reduce + latencies)
     :percentile-digest (reduce tdigest/tdigest-add
                                (tdigest/new-tdigest 100.0)
                                latencies)}))



(spec/fdef stats
           :args (spec/cat :events :reporting/events)
           :ret :reporting/aggregated-event
           )

(stest/instrument `stats)
(stest/unstrument `stats)

(defn map-vals [f m]
  (into {} (map (fn [[k v]] [k (f v)]) m)))

(defn index-events [events]
  (into {} (map (juxt :prefix identity) events)))

(defn enrich-session [indexed-events]
  (let [
        ;; REVIEW Can these orderings calculated in advance instead?
        childs-first (reverse (sort (keys indexed-events)))
        childs-tree (reduce (fn [acc child]
                              (if-let [parent (seq (butlast child))]
                                (update acc parent (fnil conj #{}) child)
                                acc))
                            {}
                            childs-first)]
    (reduce
     (fn [acc k]
       (update acc k
               (fn [x]
                 (assoc x :absolute-end
                        (if-let [childs (seq (get childs-tree k))]
                          (reduce max (map #(get-in acc [% :absolute-end]) childs))
                          (:end x))))))
     indexed-events
     childs-first)))

(defn combine-partitions [partitions]
  (let [partitions (map (fn [session]
                          ;; REVIEW we could incorporate enrich-session here
                          (map-vals vector (index-events session)))
                        partitions)
        sessions (apply merge-with concat partitions)]
    (map-vals stats sessions)))

(defn reporting-aggregation [{:keys [buffer-size sentinel]}]
  (comp
    (xf-partition-with-flush buffer-size sentinel)
    (map combine-partitions)))

(defn windowed
  "Returns a window of the passing values, newest first"
  [window-size initial-value]
  (fn [xf]
    (let [prev (volatile! (list initial-value))]
      (fn
        ([] (xf))
        ([result] (xf result))
        ([result input]
         (let [current (take window-size (cons input @prev))]
           (vreset! prev current)
           (xf result current)))))))

(comment
  (into [] (windowed 4 0) (range 5 10)))

(defn merge-aggregated-events [aggregated-events]
  (let [acc-count (reduce + (map :count aggregated-events))
        acc-counts (apply merge-with + (map :counts aggregated-events))
        acc-tdigest (tdigest/combine-tdigests (map :percentile-digest aggregated-events))
        acc-latencies (reduce + (keep :sum-latency-in-nano aggregated-events))]
    (merge (first aggregated-events)
           {:sum-latency-in-nano acc-latencies
            :count acc-count :counts acc-counts :percentile-digest acc-tdigest})))

(defn merge-prefixed-events [prefixed]
  (->> prefixed
       (mapcat identity)
       (group-by first)
       (map-vals (comp merge-aggregated-events
                       (partial map second)))))

(spec/fdef merge-windowed-stats
           :args (spec/cat :events :reporting/aggregated-events)
           :ret :reporting/aggregated-event)

(stest/instrument `merge-aggregated-events)
(stest/unstrument `merge-aggregated-events)

(defn windowed-events [window-size]
  (comp
   (windowed window-size {})
   (map merge-prefixed-events)))
