(ns coendou.registry
  (:require  [coendou.protocols :as prot]
             [coendou
              [queue :as queue]
              [semaphore :as semaphore]
              [commands :as commands]
              [core :as coendou]]))

(defprotocol IRegistry
  (registry-get [_ path])
  (registry-set!  [_ ks value])
  (registry-readable [_])
  (registry-writable [_]))

(extend-protocol IRegistry
  clojure.lang.IPersistentMap
  (registry-get [x k]
    (get x k))
  (registry-readable [x]
    (keys x))
  (registry-writable [_]
    #{})

  clojure.lang.Atom
  (registry-get [x k]
    (registry-get @x k))
  (registry-readable [x]
    (registry-readable @x))
  (registry-writable [_]
    #{})


  coendou.queue.Queue
  (registry-get [x k]
    (case k
      :in-flight (queue/in-flight x)
      :limit (queue/limit x)))

  (registry-set! [x k value]
    (case k
      :limit (queue/set-limit! x value)))

  (registry-readable [_]
    #{:in-flight :limit})

  (registry-writable [_]
    #{:limit})


  coendou.semaphore.Semaphore
  (registry-get [x k]
    (case k
      :in-flight (semaphore/in-flight x)
      :limit (semaphore/limit x)))

  (registry-set! [x k value]
    (case k
      :limit (semaphore/set-limit! x value)))

  (registry-readable [_]
    #{:in-flight :limit})

  (registry-writable [_]
    #{:limit})


  coendou.commands.Command
  (registry-get [x k]
    (case k
      ;; start-timeout-ms
      :semaphore (:semaphore x)
      :queue (:queue x)
      :start-timeout-ms (commands/start-timeout-ms x)
      :start-timeout-ns (commands/start-timeout-ns x)
      :timeout-ms (commands/timeout-ms x)))

  (registry-set! [x k value]
    (case k
      :start-timeout-ns (commands/set-start-timeout-ns x value)
      :start-timeout-ms (commands/set-start-timeout-ms x value)
      :timeout-ms       (commands/set-timeout-ms x value)))

  (registry-readable [x]
    (cond-> #{}
      (:semaphore x)        (conj :semaphore)
      (:queue x)            (conj :queue)
      (:start-timeout-ns x) (into [:start-timeout-ms :start-timeout-ns])
      (:timeout-ms x)       (conj :timeout-ms)))

  (registry-writable [x]
    (cond-> #{}
      (:start-timeout-ns x) (into [:start-timeout-ms :start-timeout-ns])
      (:timeout-ms x)       (conj :timeout-ms)))


  coendou.core.Coendou
  (registry-get [x k]
    (assert (contains? #{:commands :semaphores :queues :threadpools} k))

    @(get x k))

  (registry-set! [x k value]
    (case k
      ;; Nothing directly!
      ))

  (registry-readable [_]
    [:semaphores :queues :threadpools :commands])

  (registry-writable [_]
    #{}))

(defn get-in-registry [x ks]
  (reduce registry-get x ks))

(defn- registry-writable? [item k]
  (contains? (registry-writable item) k))

(defn set-registry! [x ks v]
  (let [item (get-in-registry x (butlast ks))
        k (last ks)]
    (if (registry-writable? item k)
      (registry-set! item k v)
      (throw (ex-info (str "Item is not writable " ks) {:item item})))))

(defn atom? [x]
  (instance? clojure.lang.Atom x))

(defn registry-snapshot [x]
  (letfn [(cached-get  [x cache path k]
            (let [x0 (registry-get x k)]
              (if-not (or (record? x0)
                          (atom? x0))
                x0
                (if-let [res (get @cache x0)]
                  res
                  (do
                    (vswap! cache assoc x0 (conj path k))
                    x0)))))

          (registry-snapshot* [cache path x]
            (if (satisfies? IRegistry x)
              (into {} (map (fn [k]
                              (let [v (cached-get x cache path k)]
                                [k (registry-snapshot* cache (conj path k) v)]))
                            (registry-readable x)))
              x))]
    (registry-snapshot* (volatile! {}) [] x)))
