(ns coendou.hystrix
  (:require [coendou.tdigest :as tdigest]
            [coendou.reporting :as reporting]))

(require '[clojure.spec :as s]
         '[clojure.future :refer :all]
         )


(s/def :command/name any?)
(s/def :command/counts (s/map-of :event/type any?))
(s/def :command/event any? ;(s/keys :req-un [:command/name :command/counts])
  )

(s/fdef hystrix-response
        :args (s/cat :event :command/event) )

(require '[clojure.spec.test :as stest])

(stest/instrument `hystrix-response)

(s/def :hystrix.command/count integer?)
(s/def :hystrix.command/counts (s/map-of keyword? integer?))
(s/def :hystrix.command/name string?)

(s/def :hystrix/message
  (s/keys :req-un
          [:hystrix.command/count
           :hystrix.command/counts
           :hystrix.command/name]))

(s/fdef hystrix-entry
        :args (s/cat :msg :hystrix/message))

(do
  (def hystrix-json [
                     "{\"currentConcurrentExecutionCount\":" :hystrix.stats/concurrent-execution-count
                     ",\"name\":" :hystrix.config/name
                     ",\"group\":" :hystrix.config/group
                     ",\"errorCount\":" :hystrix.stats/error-count
                     ",\"errorPercentage\":" :hystrix.stats/error-percentage
                     ",\"isCircuitBreakerOpen\":" :hystrix.command/circuit-breaker-open?
                     ",\"latencyExecute\":"
                     "{\"0\":"    :hystrix.latency.execute/percentile-0
                     ",\"25\":"   :hystrix.latency.execute/percentile-25
                     ",\"50\":"   :hystrix.latency.execute/percentile-50
                     ",\"75\":"   :hystrix.latency.execute/percentile-75
                     ",\"90\":"   :hystrix.latency.execute/percentile-90
                     ",\"95\":"   :hystrix.latency.execute/percentile-99
                     ",\"99\":"   :hystrix.latency.execute/percentile-99
                     ",\"99.5\":" :hystrix.latency.execute/percentile-99.5
                     ",\"100\":"  :hystrix.latency.execute/percentile-100
                     "}"
                     ",\"latencyExecute_mean\":" :hystrix.latency.execute/mean
                     ",\"latencyTotal\":"
                     "{\"0\":"    :hystrix.latency.total/percentile-0
                     ",\"25\":"   :hystrix.latency.total/percentile-25
                     ",\"50\":"   :hystrix.latency.total/percentile-50
                     ",\"75\":"   :hystrix.latency.total/percentile-75
                     ",\"90\":"   :hystrix.latency.total/percentile-90
                     ",\"95\":"   :hystrix.latency.total/percentile-95
                     ",\"99\":"   :hystrix.latency.total/percentile-99
                     ",\"99.5\":" :hystrix.latency.total/percentile-99.5
                     ",\"100\":"  :hystrix.latency.total/percentile-100
                     "}"
                     ",\"latencyTotal_mean\":" :hystrix.latency.total/mean
                     ",\"propertyValue_circuitBreakerEnabled\":" :hystrix.config.circuit-breaker/enabled?
                     ",\"propertyValue_circuitBreakerErrorThresholdPercentage\":" :hystrix.config.circuit-breaker/error-threshold-percentage
                     ",\"propertyValue_circuitBreakerForceClosed\":" :hystrix.config.circuit-breaker/force-closed?
                     ",\"propertyValue_circuitBreakerForceOpen\":"  :hystrix.config.circuit-breaker/force-open?
                     ",\"propertyValue_circuitBreakerRequestVolumeThreshold\":"    :hystrix.config.circuit-breaker/request-volume-threshold
                     ",\"propertyValue_circuitBreakerSleepWindowInMilliseconds\":" :hystrix.config.circuit-breaker/sleep-window-in-millis
                     ",\"propertyValue_executionIsolationSemaphoreMaxConcurrentRequests\":" :hystrix.config.semaphore/max-concurrent-requests
                     ",\"propertyValue_executionIsolationStrategy\":" :hystrix.config/thread-isolation-strategy
                     ",\"propertyValue_executionIsolationThreadInterruptOnTimeout\":" :hystrix.config/thread-interrupt-on-timeout?
                     ",\"propertyValue_executionIsolationThreadPoolKeyOverride\":" :hystrix.stats/thread-pool-key-override
                     ",\"propertyValue_executionIsolationThreadTimeoutInMilliseconds\":"  :hystrix.config.execution/timeout-in-millis
                     ",\"propertyValue_fallbackIsolationSemaphoreMaxConcurrentRequests\":" :hystrix.config.semaphore.fallback/max-concurrent-requests
                     ",\"propertyValue_metricsRollingStatisticalWindowInMilliseconds\":" :hystrix.stats/window-in-millis
                     ",\"propertyValue_requestCacheEnabled\":" :hystrix.config/request-caching?
                     ",\"propertyValue_requestLogEnabled\":" :hystrix.config/request-logging?
                     ",\"reportingHosts\":" :hystrix.stats/reporting-hosts
                     ",\"requestCount\":"  :hystrix.stats/request-count
                     ",\"rollingCountCollapsedRequests\":" :hystrix.stats.rolling/collapsed-requests
                     ",\"rollingCountExceptionsThrown\":" :hystrix.stats.rolling/exceptions
                     ",\"rollingCountFailure\":" :hystrix.stats.rolling-count/failure
                     ",\"rollingCountFallbackFailure\":" :hystrix.stats.rolling-count/fallback-failure
                     ",\"rollingCountFallbackRejection\":" :hystrix.stats.rolling-count/fallback-rejection
                     ",\"rollingCountFallbackSuccess\":" :hystrix.stats.rolling-count/fallback-success
                     ",\"rollingCountResponsesFromCache\":" :hystrix.stats.rolling-count/responses-from-cache
                     ",\"rollingCountSemaphoreRejected\":"  :hystrix.stats.rolling-count/semaphore-rejected
                     ",\"rollingCountShortCircuited\":" :hystrix.stats.rolling-count/short-circuited
                     ",\"rollingCountSuccess\":" :hystrix.stats.rolling-count/success
                     ",\"rollingCountThreadPoolRejected\":" :hystrix.stats.rolling-count/thread-pool-rejected
                     ",\"rollingCountTimeout\":" :hystrix.stats.rolling-count/timeout
                     ",\"type\":\"HystrixCommand\"}"])

  ;; TODO figure out how to integrate with threadpools
  (def hystrix-threadpool-json [
                     "{\"currentActiveCount\":" 0
                     ",\"name\":" :hystrix.config/name
                     ",\"currentQueueSize\":" 0
                     ",\"currentCompletedTaskCount\":" 0
                     ",\"currentCorePoolSize\":" 1
                     ",\"currentLargestPoolSize\":" 1
                     ",\"currentMaximumPoolSize\":" 1
                     ",\"currentPoolSize\":" 1
                     ",\"currentTaskCount\":" 0
                     ",\"rollingCountThreadsExecuted\":" 0
                     ",\"rollingMaxActiveThreads\":" 0
                     ",\"propertyValue_queueSizeRejectionThreshold\":" 0
                     ",\"propertyValue_metricsRollingStatisticalWindowInMilliseconds\":" (* 120 1000)
                     ",\"type\":\"HystrixThreadPool\"}"
                     ])

  (def dummy-hystrix-entry
    {

     ;; Don't see the effect in the interface
     :hystrix.latency.total/percentile-0    0.0
     :hystrix.latency.total/percentile-25   0.0
     :hystrix.latency.total/percentile-50   0.0
     :hystrix.latency.total/percentile-75   0.0
     :hystrix.latency.total/percentile-90   0.0
     :hystrix.latency.total/percentile-95   0.0
     :hystrix.latency.total/percentile-99   0.0
     :hystrix.latency.total/percentile-99.5 0.0
     :hystrix.latency.total/percentile-100  0.0


     :hystrix.config.circuit-breaker/force-closed? false
     :hystrix.config.circuit-breaker/force-open? false
     :hystrix.latency.total/mean 2.0
     :hystrix.latency.execute/mean 2.1
     :hystrix.config.circuit-breaker/enabled? false
     :hystrix.command/circuit-breaker-open? false
     :hystrix.stats/thread-pool-key-override (pr-str "?")
     :hystrix.config/thread-interrupt-on-timeout? true
     :hystrix.config/thread-isolation-strategy (pr-str "semaphore") ;; or :thread
     :hystrix.stats/reporting-hosts 1
     :hystrix.config/request-logging? false
     :hystrix.config/request-caching? false
     :hystrix.config.circuit-breaker/error-threshold-percentage 1.0
     :hystrix.config.circuit-breaker/request-volume-threshold 10
     :hystrix.config.circuit-breaker/sleep-window-in-millis 1000
     :hystrix.config.semaphore/max-concurrent-requests 0
     :hystrix.config.semaphore.fallback/max-concurrent-requests 0
     :hystrix.stats.rolling-count/failure 3
     :hystrix.stats.rolling-count/success 1
     :hystrix.stats.rolling-count/short-circuited 1
     :hystrix.stats.rolling-count/timeout 1
     :hystrix.stats/window-in-millis (* 120 1000)
     :hystrix.config.execution/timeout-in-millis 200

     :hystrix.stats/concurrent-execution-count 0
     :hystrix.stats.rolling-count/fallback-failure 0
     :hystrix.stats.rolling-count/fallback-rejection 0
     :hystrix.stats.rolling-count/fallback-success 0
     :hystrix.stats.rolling-count/responses-from-cache 0
     :hystrix.stats.rolling-count/semaphore-rejected 0
     :hystrix.stats.rolling/collapsed-requests 0
     :hystrix.stats.rolling/exceptions 0

     :hystrix.stats.rolling-count/thread-pool-rejected 0}
    )

  (let [parsed-tpl (clojure.walk/postwalk (fn [x]
                                            (if (keyword? x)
                                              (fn [m] (get m x (str "\"missing:" x \")))
                                              x))
                                          hystrix-json)]

    (defn hystrix-format [m]
      (let [m (merge dummy-hystrix-entry m)]
        (clojure.string/join nil (map (fn [t]
                                        (if (fn? t)
                                          (t m)
                                          t))
                                      parsed-tpl))))


    ))

(let [parsed-tpl (clojure.walk/postwalk (fn [x]
                                            (if (keyword? x)
                                              (fn [m] (get m x (str "\"missing:" x \")))
                                              x))
                                        hystrix-threadpool-json)]
  (defn hystrix-threadpool-format [m]
    (clojure.string/join nil (map (fn [t]
                                    (if (fn? t)
                                      (t m)
                                      t))
                                  parsed-tpl))))
(defn hystrix-messages [events]
  (map hystrix-format events))

(defn hystrix-entries [indexed-events]
  (map
   (fn [{:keys [counts] :as event command-name :cmd}]
     (let [digest (:percentile-digest event)
           latency-sum (:sum-latency-in-nano event)
           request-count (:count event)
           error-count (:error counts 0)
           error-percentage (if (zero? request-count)
                              0.0
                              (* 100.0 (/ error-count request-count)))
           ;; Quantiles are in nanoseconds need to make ms
           nano->ms (fn [x] (/ x 1000 1000.0))
           quantile (fn [qtl] (nano->ms (tdigest/quantile digest qtl)))
           mean (if (zero? request-count)
                  0.0
                  (nano->ms (/ latency-sum request-count)))]
       {:hystrix.config/name (pr-str (str command-name))
        :hystrix.config/group (pr-str (str command-name " group"))

        :hystrix.latency.total/mean 0.0
        :hystrix.latency.execute/mean mean

        :hystrix.latency.execute/percentile-0    (quantile 0.0)
        :hystrix.latency.execute/percentile-25   (quantile  0.25)
        :hystrix.latency.execute/percentile-50   (quantile  0.5)
        :hystrix.latency.execute/percentile-75   (quantile  0.75)
        :hystrix.latency.execute/percentile-90   (quantile  0.9)
        :hystrix.latency.execute/percentile-95   (quantile  0.95)
        :hystrix.latency.execute/percentile-99   (quantile  0.99)
        :hystrix.latency.execute/percentile-99.5 (quantile  0.995)
        :hystrix.latency.execute/percentile-100  (quantile  1.0)

        :hystrix.stats/request-count request-count
        :hystrix.stats/error-count error-count
        :hystrix.stats/error-percentage error-percentage
        :hystrix.stats.rolling-count/failure error-count
        :hystrix.stats.rolling-count/success (:success counts 0)
        :hystrix.stats.rolling-count/short-circuited (:short-circuited counts 0)
        :hystrix.stats.rolling-count/timeout (+ (:timeout counts 0)
                                                (:start-timeout counts 0))


        }))
   (or (vals indexed-events) [])))

(defn xf-hystrix-data []
   (comp
    (reporting/windowed-events 120)
    (map #(hystrix-entries %))))

(defn xf-hystrix-format []
  (comp
   (xf-hystrix-data)
   (map #(hystrix-messages %))))

(require '[manifold.stream :as stream])

(let [timeout-val (Object.)
      drained-val (Object.)]
  (defn take-available! [s]
    (loop [acc []]
      (let [ret @(stream/try-take! s drained-val 0 timeout-val)]
        (if (or (= ret timeout-val)
                (= ret drained-val))
          acc
          (recur (conj acc ret)))))))

(require '[manifold.time :as t])

(defn periodically-take-available! [source period-ms f]
  (t/every period-ms #(f (take-available! source))))

(defn sse-message [m]
  (str "data: " m "\n\n"))

(defn hystrix-stream* [session-stream {:keys [timeout-ms buffer-size]}]
  (let [sentinel (Object.)
        aggregate-stream (stream/transform
                          (reporting/reporting-aggregation {:buffer-size buffer-size :sentinel ::sentinel})
                          session-stream)
        batch-stream (stream/stream 1000)
        output-stream (stream/transform
                       (comp
                        (xf-hystrix-format)
                        ;; REVIEW preferably we would use mapcat here, but that doesn't seem to flush correctly
                        (map (fn [data]
                               (if (seq data)
                                 (clojure.string/join nil (map sse-message data))
                                 (sse-message "{}")))))
                       batch-stream)
        cancel-fn (periodically-take-available! aggregate-stream timeout-ms (fn [batch]
                                                                              (stream/put! batch-stream (reporting/merge-prefixed-events batch))
                                                                              ;; Make sure batches will be flushed
                                                                              (stream/put! session-stream ::sentinel)))
        cancel-streams (fn []
                         (cancel-fn)

                         (stream/close! aggregate-stream)
                         (stream/close! batch-stream))]

    (stream/on-closed session-stream cancel-streams)
    (stream/on-closed batch-stream cancel-streams)
    output-stream))

(defn hystrix-stream [subscription-stream]
  (let [session-stream (stream/stream 1000)
        _ (stream/connect
           subscription-stream
           session-stream)]
    (hystrix-stream* session-stream {:timeout-ms 1000 :buffer-size 1000})))
