(ns coendou.core
  (:require
   [manifold
    [deferred :as d]
    [stream :as s]]
   [coendou
    [protocols :as prot]
    [commands :as commands]
    [semaphore :as semaphore]
    [queue :as queue]]
   [clojure.spec :as spec]
   [clojure.spec.test :as stest]
   [coendou.reporting :as reporting]
   [manifold.bus :as bus]))

(defn passport-enter [passport-value]
  (update passport-value :in-flight inc))

(defn passport-leave [passport-value msg]
  (-> passport-value
      (update :in-flight dec)
      (update :stamps conj msg)))

(defn passport-control!
  "Collect metrics of all commands in the current session"
  [passport completion-fn [type msg]]
  (let [process (if (= type :start)
                  passport-enter
                  #(passport-leave % msg))
        {:keys [in-flight stamps] :as new-passport}
        (swap! passport process)]
    (let [done? (cond
                  (pos? in-flight) false
                  (zero? in-flight) true
                  :else (throw (ex-info "Missing a stamp!" {:passport new-passport})))]
      (when done?
        (completion-fn stamps)))))

(defrecord MockedContext []
  prot/IReport
  (report-execution [_ type msg]
    :noop)

  prot/IContext
  (get-context [this]
    this)

  prot/ICommand
  (execute-command [ctx cmd args]
    (prot/execute-command cmd ctx args)))

(defrecord Context [prefix passport publish-fn]
  prot/IReport
  (report-execution [_ type msg]
    (passport-control! passport publish-fn [type msg]))

  prot/IContext
  (get-context [this]
    this)

  prot/ICommand
  (execute-command [ctx cmd args]
    (prot/execute-command cmd ctx args)))

(defn create-passport []
  (atom {:in-flight 0 :stamps []}))

(defn publish-session [coendou session]
  (bus/publish! (get coendou :event-bus) :sessions session))

(defn gen-context [completion-fn]
  (->Context []
             (create-passport)
             completion-fn))

(defrecord Coendou [event-bus
                    commands
                    semaphores
                    queues
                    threadpools
                    ]
  prot/ILifecycle
  (start [component]
    (let [subscriber-buffer-size 1000
          event-bus (bus/event-bus #(s/stream subscriber-buffer-size))]
      (assoc component
             :event-bus event-bus
             :commands (atom {})
             :semaphores (atom {})
             :queues (atom {})
             :threadpools (atom {}))))

  (stop [component]
    (assoc component
           :event-bus nil
           :commands nil
           :semaphores nil
           :queues nil
           :threadpools nil))

  prot/IContext
  (get-context [this]
    (gen-context
     (fn [session]
       (publish-session this session))))

  prot/ISubscribeReport
  (report-subscription [_ topic]
    ;; No timeout here
    (bus/subscribe event-bus topic)))

;; Public API

(defn create [x]
  (map->Coendou x))

(defn coendou? [x]
  (instance? Coendou x))

(defn start [coendou]
  (prot/start coendou))

(defn mocked-coendou []
  (start (create {})))

(defn stop [coendou]
  (prot/stop coendou))

(defn register-semaphore! [coendou semaphore-name limit]
  (swap! (:semaphores coendou) assoc semaphore-name (semaphore/create limit)))

(defn get-entity [coendou type entity-name]
    (let [type-values @(get coendou type)]
    (or (get type-values entity-name)
        (throw (ex-info (str "Could not find " entity-name " in " (name type)) {type type-values})))))

(defn get-semaphore [coendou name]
  (get-entity coendou :semaphores name))

(defn register-queue! [coendou queue-name limit]
  (swap! (:queues coendou) assoc queue-name (queue/create limit)))

(defn register-command! [coendou command-id command]
  (swap! (:commands coendou) assoc command-id command))

(defn get-queue [coendou name]
  (get-entity coendou :queues name))

(defn report-subscription [coendou topic]
  (prot/report-subscription coendou topic))

(defn context-with [ctx value-map]
  (merge (prot/get-context ctx) value-map))

(defn execute-command [ctx cmd args]
  (let [nested-ctx (prot/get-context ctx)]
    (prot/execute-command nested-ctx cmd args)))

(defn create-fallback [fn-or-value]
  (when fn-or-value
    (if (fn? fn-or-value)
      fn-or-value
      (constantly fn-or-value))))

(defn create-command [coendou {command-id :id
                               f :fn
                               :keys [start-timeout-ms
                                      semaphore-name
                                      queue-name
                                      timeout-ms
                                      deferred-fn
                                      fallback-value
                                      fallback
                                      error-fallback-value
                                      error-fallback
                                      timeout-fallback-value
                                      timeout-fallback
                                      start-timeout-fallback-value
                                      start-timeout-fallback
                                      queue-rejected-fallback-value
                                      queue-rejected-fallback]
                               :as config}]
  (let [action (if deferred-fn
                 (commands/deferred-fn deferred-fn)
                 (commands/sync-fn f))]
    (let [fallback (create-fallback (or fallback fallback-value))
          error-fallback (create-fallback (or error-fallback error-fallback-value))
          start-timeout-fallback (create-fallback (or start-timeout-fallback start-timeout-fallback-value))
          timeout-fallback (create-fallback (or timeout-fallback timeout-fallback-value))
          queue-rejected-fallback (create-fallback (or queue-rejected-fallback queue-rejected-fallback-value))

          config0 (into {}
                        (filter val
                                (-> (assoc config
                                           :action-fn action
                                           ;; We should have one way of defining fallbacks!
                                           :fallback fallback
                                           :error-fallback error-fallback
                                           :start-timeout-fallback start-timeout-fallback
                                           :timeout-fallback timeout-fallback
                                           :queue-rejected-fallback queue-rejected-fallback)
                                    (cond->
                                        start-timeout-ms
                                      (assoc :start-timeout-ns (commands/ms->nanos start-timeout-ms))

                                      queue-name
                                      (assoc :queue (get-queue coendou queue-name))

                                      semaphore-name
                                      (assoc :semaphore (get-semaphore coendou semaphore-name)))
                                    (update :start-timeout-ns (fn [x] (when x (atom x))))
                                    (update :timeout-ms (fn [x] (when x (atom x))))
                                    )))
          cmd (commands/create config0)]
      (register-command! coendou command-id cmd)
      cmd)))

(defn context? [x]
  (satisfies? prot/IContext x))

(spec/fdef create-command
           :args (spec/cat
                  ;; Mock coendou implements Context protocol not, coendou
                  :coendou #_coendou? context?
                  :config :coendou/command))
(stest/instrument `create-command)
