(ns coendou.commands
  (:require [manifold.deferred :as d]
            [clojure.spec :as spec]
            [clojure.spec.test :as stest]
            [clojure.future :refer :all]
            [coendou.specs]
            [coendou
             [queue :as queue]
             [semaphore :as semaphore]]))

;; TODO use future-with to choose a thread pool

(defn deferred-fn [f]
  (fn [ctx args]
    (d/chain (f ctx args)
             (fn [x]
               [:success x]))))

(spec/fdef deferred-fn
           :args (spec/cat :fn :coendou/fn) )
(stest/instrument `deferred-fn)

(defn sync-fn [f]
  (fn [ctx args]
    (d/future [:success (f ctx args)])))

(spec/fdef sync-fn
           :args (spec/cat :fn :coendou/fn) )
(stest/instrument `sync-fn)

(defn with-error-handling [action-fn {:keys [error-handler
                                             error-fallback]}]
  (fn [ctx args]
    (let [ref (d/deferred)]
      (d/on-realized (action-fn ctx args)
                     (fn [x]
                       (d/success! ref x))
                     (fn [ex]
                       ;; Wrapping the error handler with yes, another error handler
                       ;; so we release the deferred in case the unexpected happens!
                       (try
                         (error-handler ex)
                         (d/success! ref [:error (error-fallback ctx args)])
                         (catch Throwable ex
                           (d/error! ref (ex-info (str "Error in callback: " (.getMessage ex)) {} ex))))))
      ref)))

(spec/fdef with-start-timeout :args (spec/cat :action-fn :coendou/action-fn :config :command.part/error-handling))
(stest/instrument `with-start-timeout)


(defn with-start-timeout [action-fn {:keys [start-timeout-fallback
                                            nano-time-fn
                                            start-timeout-ns]}]
  (fn [ctx args]
    (let [start (get ctx :start)]
      (let [elapsed-ns (- (nano-time-fn) start)]
        (if (< elapsed-ns @start-timeout-ns)
          (action-fn ctx args)
          (d/success-deferred [:start-timeout (start-timeout-fallback ctx args)]))))))

(spec/fdef with-start-timeout :args (spec/cat :action-fn :coendou/action-fn :config :command.part/start-timeout))
(stest/instrument `with-start-timeout)


(require '[coendou.protocols :as prot])
(require '[manifold.stream :as s])


(defn with-semaphore [action-fn {:keys [semaphore]}]
  (fn [ctx args]
    (-> (d/chain (semaphore/acquire! semaphore)
                 (fn [_] (action-fn ctx args)))
        (d/finally
          (fn [] (semaphore/release! semaphore))))))

(spec/fdef with-semaphore :args (spec/cat :action-fn :coendou/action-fn :config :command.part/semaphore))
(stest/instrument `with-semaphore)


(defn with-semaphore-priority [action-fn {:keys [semaphore]}]
  (fn [ctx args]
    ;; Acquire but don't wait
    (semaphore/acquire! semaphore)
    (-> (action-fn ctx args)
        (d/finally
          (fn [] (semaphore/release! semaphore))))))

(spec/fdef with-semaphore-priority :args (spec/cat :action-fn :coendou/action-fn :config :command.part/semaphore-priority))
(stest/instrument `with-semaphore-priority)


(let [sentinel (Object.)]
  (defn with-timeout [action-fn {:keys [timeout-ms
                                        timeout-fallback]}]
    (fn [ctx args]
      (d/chain (d/timeout! (action-fn ctx args) @timeout-ms sentinel)
               (fn [x]
                 (if (= sentinel x)
                   [:timeout (timeout-fallback ctx args)]
                   x))))))

(spec/fdef with-timeout :args (spec/cat :action-fn :coendou/action-fn :config :command.part/timeout))
(stest/instrument `with-timeout)


(defn with-queue [action-fn {:keys [queue-rejected-fallback
                                    queue]}]
  (fn [ctx args]
    (if (queue/queue! queue)
      (-> (action-fn ctx args)
          (d/finally
            (fn [] (queue/dequeue! queue))))
      (d/success-deferred [:rejected (queue-rejected-fallback ctx args)]))))

(spec/fdef with-queue :args (spec/cat :action-fn :coendou/action-fn :config :command.part/queue))
(stest/instrument `with-queue)


(defn with-queue-priority [action-fn {:keys [queue]}]
  (fn [ctx args]
    (queue/unsafe-queue! queue)
    (-> (action-fn ctx args)
        (d/finally
          (fn [] (queue/dequeue! queue))))))

(spec/fdef with-queue-priority :args (spec/cat :action-fn :coendou/action-fn :config :command.part/queue-priority))
(stest/instrument `with-queue-priority)


(defn with-reporting [action-fn {command-id :id :keys [nano-time-fn ms-time-fn]}]
  (fn [ctx args]
    (let [start-nanos (nano-time-fn)
          start-time (ms-time-fn)
          ctx (update ctx :prefix conj command-id)
          _ (prot/report-execution ctx :start nil)
          report (fn [status]
                   (let [end-nanos (nano-time-fn)
                         elapsed (- end-nanos start-nanos)]
                     (prot/report-execution ctx :end {:prefix (get ctx :prefix)
                                                      :id command-id
                                                      :type status
                                                      :start start-time
                                                      :start-nanos start-nanos
                                                      :end-nanos end-nanos
                                                      :elapsed elapsed})))]
      (d/on-realized (action-fn ctx args)
                     (fn [[status _]]
                       (report status))
                     (fn [ex]
                       (report :error))))))

(spec/fdef with-reporting :args (spec/cat :action-fn :coendou/action-fn :config :command.part/reporting))
(stest/instrument `with-reporting)


(defn nano-time []
  (System/nanoTime))

(defn ms->nanos [ms]
  (long (* ms 1000 1000)))

(defn nanos->ms [nanos]
  (double (/ nanos 1000 1000)))

(defn start-timeout-ns [x]
  (when-let [start-timeout-ns (:start-timeout-ns x)]
    @start-timeout-ns))

(defn start-timeout-ms [x]
  (when-let [start-timeout-ns (:start-timeout-ns x)]
    (nanos->ms @start-timeout-ns)))

(defn set-start-timeout-ns [x value]
  (reset! (:start-timeout-ns x) value))

(defn set-start-timeout-ms [x value]
  (reset! (:start-timeout-ns x) (ms->nanos value)))

(defn timeout-ms [x]
  (when-let [timeout-ms (:timeout-ms x)]
    @timeout-ms))

(defn set-timeout-ms [x value]
  (reset! (:timeout-ms x) value))

;; REVIEW Temp hack to get command tests working
(extend-protocol prot/IReport
  clojure.lang.IPersistentMap
  (report-execution [_ _ _]
    :noop))

(defn execute-command* [name command-fn ctx args]
  (d/chain (command-fn ctx args)
           (fn [[_ value]]
             value)))

(defrecord Command [name
                    action-fn
                    start-timeout-ns
                    priority?
                    semaphore
                    timeout-ms
                    queue]
  prot/ICommand
  (execute-command [cmd ctx args]
    (execute-command* name action-fn ctx args)))

(defn execute-command [cmd ctx args]
  (prot/execute-command cmd ctx args))

(defn atom? [x]
  (instance? clojure.lang.Atom x))

(defn ->atom [x]
  (if (atom? x)
    x
    (atom x)))

(defn- wrap [{:keys [action-fn] :as config} with-fn command-part-config]
  (assoc (merge config command-part-config)
         :action-fn (with-fn action-fn command-part-config)))

(defn command-map* [{:keys [action-fn
                            start-timeout-ns
                            timeout-ms
                            fallback
                            fallback-value
                            priority?
                            semaphore
                            queue] :as config}]
  (let [fallback (or fallback (constantly fallback-value))]
    (cond-> config
      true ;; always right?
      (wrap with-error-handling
            (update config :error-fallback (fnil identity fallback)))

      ;; Start timeout just before starting
      start-timeout-ns
      (wrap with-start-timeout
             (-> config
                 (update :start-timeout-ns ->atom)
                 (update :start-timeout-fallback (fnil identity fallback))))

      ;; Semaphores just before starting protecting the concurrency
      (and semaphore priority?)
      (wrap with-semaphore-priority config)

      (and semaphore (not priority?))
      (wrap with-semaphore config)

      ;; Timeout's around almost at the end
      timeout-ms
      (wrap with-timeout
             (-> config
                 (update :timeout-ms ->atom)
                 (update :timeout-fallback (fnil identity fallback))))

      ;; Queue's at the end i.e. the beginning of the chain
      (and priority? queue)
      (wrap with-queue-priority config)

      (and (not priority?) queue)
      (wrap with-queue
            (update config :queue-rejected-fallback (fnil identity fallback)))

      true ;; we currently always want reporting, but might change?
      (wrap with-reporting
            (->
             (select-keys config [:id :nano-time-fn])
             (assoc :ms-time-fn #(System/currentTimeMillis)))))))

(defn create [config]
  (let [defaults {:nano-time-fn nano-time}
        command-config (command-map* (merge defaults config))]
    (map->Command command-config)))

(spec/fdef create
           :args (spec/cat :config :coendou.command/command))
(stest/instrument `create)
