(ns coendou.queue
  (:require [coendou.protocols :as prot]
            [clojure.spec :as spec]
            [clojure.spec.test :as stest]))

(defrecord Queue [in-flight limit-state]
  prot/IQueue
  (unsafe-queue! [_]
    (swap! in-flight dec))

  (queue! [_]
    (let [success? (volatile! nil)]
      (swap! in-flight (fn [x]
                         (vreset! success? false)
                         (let [x0 (dec x)]
                           (if (neg? x0)
                             x
                             (do
                               (vreset! success? true)
                               x0)))))
      @success?))

  (dequeue! [_]
    (swap! in-flight inc)))

(defn unsafe-queue! [q]
  (prot/unsafe-queue! q))

(defn queue! [q]
  (prot/queue! q))

(defn dequeue! [q]
  (prot/dequeue! q))

(defn change-limit! [q delta]
  (prot/change-limit! q delta))

(defn queue? [x]
  (instance? Queue x))

(defn limit [x]
  @(:limit-state x))

(defn in-flight [x]
  (- (limit x) @(:in-flight x)))

(defn- set-limit!* [limit-state new-limit]
  (assert (>= new-limit 0) (str "Limit should be greater than or equal to zero"))
  (let [old-limit @limit-state]
    (if (compare-and-set! limit-state old-limit new-limit)
      (- new-limit old-limit)
      (throw (ex-info "Concurrent limit update detected" {})))))

(defn set-limit! [x value]
   (let [delta (set-limit!* (:limit-state x) value)]
     (swap! (:in-flight x) + delta)))

(defn create [initial-limit]
  (->Queue (atom initial-limit) (atom initial-limit)))
