(ns coendou.rolling-stats
  (:require [amalloy.ring-buffer :refer [ring-buffer]])
  (:import [com.tdunning.math.stats TDigest]))

(defprotocol INextBucket
  (next-bucket [_ bucket-index]))

(defprotocol IIncrement
  (increment [this bucket-index]))

(defn snapshot [this bucket-index]
  (next-bucket this bucket-index))

(defprotocol ISum
  (sum [_]))

(defrecord RollingCount [previous-buckets current-bucket bucket-index]
  INextBucket
  (next-bucket [this desired-bucket-index]
    (if (<= desired-bucket-index bucket-index)
      this
      (next-bucket
       (RollingCount. (conj previous-buckets current-bucket) 0 (inc bucket-index))
       desired-bucket-index)))

  ISum
  (sum [_]
    (reduce + current-bucket previous-buckets))

  IIncrement
  (increment [this current-bucket-index]
    (update (next-bucket this current-bucket-index) :current-bucket inc)))

(defn rolling-count [number-of-buckets bucket-index]
  (->RollingCount (ring-buffer (dec number-of-buckets)) 0 bucket-index))

(defprotocol IAddValue
  (add-value [_ value bucket-index]))

(defrecord RollingSum [previous-buckets current-bucket bucket-index]
  INextBucket
  (next-bucket [this desired-bucket-index]
    (if (<= desired-bucket-index bucket-index)
      this
      (next-bucket
       (RollingSum. (conj previous-buckets current-bucket) 0 (inc bucket-index))
       desired-bucket-index)))

  ISum
  (sum [_]
    (reduce + current-bucket previous-buckets))

  IAddValue
  (add-value [this value current-bucket-index]
    (update (next-bucket this current-bucket-index) :current-bucket + value)))

(defn rolling-sum [{:keys [number-of-buckets bucket-index]}]
  (->RollingSum (ring-buffer (dec number-of-buckets)) 0 bucket-index))

(defn new-tdigest [accuracy]
  (TDigest/createDigest 100.0))

(defn tdigest-add [tdigest value]
  (.add tdigest (double value))
  tdigest)

(defn merge-tdigests [tdigest-base tdigest-other]
  (.add tdigest-base tdigest-other)
  tdigest-base)

(defn quantile [tdigest q]
  (.quantile tdigest q))

(defrecord RollingQuantile [previous-buckets current-bucket bucket-index accuracy]
  INextBucket
  (next-bucket [this desired-bucket-index]
    (if (<= desired-bucket-index bucket-index)
      this
      (next-bucket
       (RollingQuantile. (conj previous-buckets current-bucket) (new-tdigest accuracy) (inc bucket-index) accuracy)
       desired-bucket-index)))

  ISum
  (sum [_]
    (reduce merge-tdigests
            (new-tdigest accuracy)
            (cons current-bucket previous-buckets)))
  IAddValue
  (add-value [this value current-bucket-index]
    (update (next-bucket this current-bucket-index) :current-bucket tdigest-add value)))

(defn rolling-counts [number-of-buckets current-bucket-index]
  (let [num (rolling-count number-of-buckets current-bucket-index)]
    {:count num
     :success num
     :timeout num
     :failed num
     :short-circuited num}))

(defn rolling-quantile [{:keys [number-of-buckets bucket-index accuracy]}]
  (map->RollingQuantile
   {:previous-buckets  (ring-buffer (dec number-of-buckets))
    :current-bucket (new-tdigest accuracy)
    :bucket-index bucket-index
    :accuracy 100.0}))



(defn rolling-counts->snapshot [numbers current-bucket-index]
  (into {} (map (fn [[k v]]
                  [k (snapshot v current-bucket-index)])
                numbers)))

(defn rolling-counts->sum [numbers current-bucket-index]
  (into {} (map (fn [[k v]]
                  [k (sum (snapshot v current-bucket-index))])
                numbers)))

(defn update-rolling-counts [numbers event current-bucket-index]
  (->
   numbers
   (update :count increment current-bucket-index)
   (update event increment current-bucket-index)))
