(ns coendou.core
  (:require [clojure.core.async :as a :refer [go <! >! chan]]
            [coendou.rolling-stats :as stats]
            [amalloy.ring-buffer :refer [ring-buffer]]))

(defprotocol ICommand
  (-execute-command [_ cmd args]))

(defn fn->async-fn [cmd]
  (let [f (:fn cmd)]
    (fn [ctx args]
      (a/go
        (try
          [:success (apply f ctx args)]
          (catch AssertionError e
            [:failed e])
          (catch Exception e
            [:failed e]))))))

(defrecord Command [id name async-fn fallback-cmd timeout-ms
                    rate-limit rate-agent
                    short-circuit-limit]
  ICommand
  (-execute-command [this ctx args]
    (async-fn ctx args)))

(defn static-chan [v]
  (let [ch (a/promise-chan)]
    (a/put! ch v)
    ch))

(defn timeout-command-config [{:keys [async-fn timeout-ms] :as cmd-config}]
  (if timeout-ms
    (assoc cmd-config
      :async-fn (fn timeout [ctx args]
                  (go
                    (let [timeout-ch (a/timeout timeout-ms)
                          cmd-ch (async-fn ctx args)
                          [v ch] (a/alts! [cmd-ch timeout-ch])]
                      (if (= ch timeout-ch)
                        [:timeout :coendou/timeout]
                        v)))))
    cmd-config))

(defn rate-limited-config [{:keys [rate-limit async-fn] :as cmd-config}]
  (if rate-limit
    (let [rate-limited-result (static-chan [:rate-limited :coendou/rate-limited])
          rate-agent #_(atom rate-limit) (agent rate-limit)
          ]
      (assoc cmd-config
        :rate-agent rate-agent
        :async-fn (fn rate-limit [ctx args]
                    (if (> @rate-agent 0)
                      (do
                        (send rate-agent dec)
                        ;(swap! rate-agent dec)
                        (async-fn ctx args))
                      rate-limited-result))))
    cmd-config))

(defn dec-or-nil [counter]
  (when counter
    (dec counter)))

;; Flow chart https://raw.githubusercontent.com/wiki/Netflix/Coendou/images/coendou-command-flow-chart.png
;; TODO implement circuit breaker open. Read coendou stats and see if we have a go
(defn short-circuit-config [{:keys [async-fn] :as cmd-config}]
  (let [short-circuit-result (static-chan [:short-circuited :coendou/short-circuited])
        short-circuit-limit (atom nil)]
    (assoc cmd-config
      :short-circuit-limit short-circuit-limit
      :async-fn (fn short-circuit [ctx args]
                  (if-let [limit @short-circuit-limit]
                    (if (> limit 0)
                      (do
                        (swap! short-circuit-limit dec-or-nil)
                        (async-fn ctx args))
                      short-circuit-result)
                    (async-fn ctx args))))))

(defn create-fallback-command [config]
  (map->Command config))

;; https://github.com/Netflix/Coendou/wiki/Configuration#fallbackisolationsemaphoremaxconcurrentrequests
(defn concurrency-limit-config [{:keys [concurrency-limit async-fn] :as cmd-config}]
  (let [rejected-result (static-chan [:rejected :coendou/rejected])
        concurrency-counter (atom concurrency-limit)]
    (assoc cmd-config
      :concurrency-counter concurrency-counter
      :async-fn (fn concurrency-control [ctx args]
                  (if (> @concurrency-counter 0)
                    (go
                      (swap! concurrency-counter dec)
                      (let [res (<! (async-fn ctx args))]
                        (swap! concurrency-counter inc)
                        res))
                    rejected-result)))))

(defn create-command* [config]
  (let [config (cond->
                (-> config
                    (assoc :id (Object.))
                    (clojure.set/rename-keys {:fallback :fallback-cmd}))

                ;; Set timeout to 100ms by default
                (= (get config :timeout-ms :not-found) :not-found)
                (assoc :timeout-ms 100)

                ;; Set timeout to 100ms by default
                (= (get config :concurrency-limit :not-found) :not-found)
                (assoc :concurrency-limit 2))]
    (-> config
        timeout-command-config
        rate-limited-config
        short-circuit-config
        concurrency-limit-config
        map->Command)))

(defn report-event
  ([channel prefix cmd result start]
   (report-event channel prefix cmd result start (System/nanoTime)))
  ([channel prefix cmd result start end]
   (a/put! channel [:end {:start start :end end :prefix prefix :event result :cmd (:name cmd)
                          :id (:id cmd)}])))

(defn -execute-command* [ctx cmd args]
  (go
    (let [start (System/nanoTime)
          [type _ :as result] (<! (-execute-command cmd ctx args))]
      (report-event (:session-channel ctx) (conj (:prefix ctx) (name type)) cmd type start)
      (if (= :success type)
        result
        ;; REVIEW we can also do this check at compile time by having two functions
        (if-let [fallback-cmd (:fallback-cmd cmd)]
          (<! (-execute-command (update ctx :prefix conj "fallback") fallback-cmd args))
          result)))))

(defprotocol IContext
  (get-context [_]))

(defrecord Context [prefix session-channel]
  IContext
  (get-context [this]
    this)

  ICommand
  (-execute-command [this cmd args]
    (a/put! session-channel [:start])
    (if prefix
      (-execute-command* (assoc this :prefix (conj prefix (:name cmd))) cmd args)
      (go
        (let [prefix [(:name cmd)]
              start (System/nanoTime)
              res (<! (-execute-command* (assoc this :prefix prefix) cmd args))]
          (report-event session-channel prefix (assoc cmd :id nil) (first res) start)
          res)
        ))))

(defn gen-context* [report-channel]
  ;; Run reporting channel until all command have succeed
  (let [session-channel (a/chan 20)
        ctx (->Context nil session-channel)]
    (go
      (let [start (System/currentTimeMillis)
            res (loop [count 1
                       acc {:cmds [] :start Long/MAX_VALUE :end Long/MIN_VALUE }]
                  (let [[type msg] (<! session-channel)
                        [new-count new-acc] (condp = type
                                              :start [(inc count) acc]
                                              :end   [(dec count) (-> acc
                                                                    (update :start min (:start msg))
                                                                    (update :end   max (:end msg))
                                                                    (update :cmds  conj msg))]
                                              (throw (ex-info "No such type" {})))]
                    (if (zero? new-count)
                      new-acc
                      (recur new-count new-acc))))]
        (a/close! session-channel)

        (a/>! report-channel res)))

    ctx))


(defn update-counts [counts event]
  (-> counts
      (update :count (fnil inc 0))
      (update event  (fnil inc 0))))

(defn bucket-xf [f]
  (comp
   (map (fn [msg] (assoc msg :bucket (f msg))))
   (partition-by f)))

(defn bucket-stats [grouped]
  (into {} (map (fn [[prefix vs]]
                  (let [counts (reduce update-counts {} (map :event vs))
                        stats (assoc counts :elapsed (reduce stats/tdigest-add
                                                             (stats/new-tdigest 100.0)
                                                             (map (fn [{:keys [start end]}] (- end start))
                                                                  vs)))]
                    [prefix stats])))
        (group-by :prefix (mapcat :cmds grouped))))

(defn merge-buckets [buckets]
  (apply merge-with (fn [& args]
                      (assoc
                          (apply merge-with + (map (fn [x] (dissoc x :elapsed)) args))
                        :elapsed (reduce stats/merge-tdigests (map :elapsed args))))
         buckets))

(defn get-cmd-stats [grouped]
  (into {} (map (fn [[id vs]]
                  (let [counts (reduce update-counts {} (map :event vs))
                        stats (assoc counts :elapsed (reduce stats/tdigest-add
                                                             (stats/new-tdigest 100.0)
                                                             (map (fn [{:keys [start end]}] (- end start))
                                                                  vs)))]
                    [id stats])))
        (group-by :id (mapcat :cmds grouped))))

(defprotocol ICreateCommand
  (create-command [_ cmd-config]))

(defprotocol IDestroyCommand
  (destroy-command [_ cmd-config]))

(defprotocol IReportSubcription
  (report-subscription [_]))

(def default-bucket-step-fn
  (fn [_]
    (long (/ (System/currentTimeMillis) 1000.0))))


(defn reporting-loop [{:keys [switch input-channel output-channel bucket-step-fn
                              window-size
                              cmd-state]}]
  ;; TODO should we run the below in a seperate thread?
  (a/go
    (loop [prefix-buckets (ring-buffer window-size)
           cmds-buckets   (ring-buffer window-size)]
      (when @switch
        (let [start (System/currentTimeMillis)
              grouped (a/<! input-channel)
              all-cmds (map (comp :id :command) (mapcat :cmds grouped))

              prefix-bucket (bucket-stats grouped)
              new-prefix-buckets (conj prefix-buckets prefix-bucket)
              windowed-prefix-stats (merge-buckets new-prefix-buckets)

              cmds-bucket (get-cmd-stats grouped)
              new-cmds-buckets (conj cmds-buckets cmds-bucket)
              windowed-cmds-stats (merge-buckets new-cmds-buckets)]

          ;; Do something per command
          (doseq [[id cmd] @cmd-state]
            (when-let [rate-limit (:rate-limit cmd)]
              (send (:rate-agent cmd) + rate-limit))

            (when-let [cmd-bucket (get cmds-bucket id)]
              (when-let [short-circuit-fn (:short-circuit-fn cmd)]
                (reset! (:short-circuit-limit cmd)
                        (short-circuit-fn cmd-bucket (get windowed-cmds-stats id))))))

          (a/>! output-channel {:topic :buckets, :msg windowed-prefix-stats})

          (recur new-prefix-buckets new-cmds-buckets))
        ))))

;; execute-command puts something on the report-channel
;; if we manually get this from the report-channel and move it to the grouping channel
;; When we increase the bucketing function we can wait for the grouping channel, or for the publish channel

(defprotocol ICoendouLifecycle
  (start [_])
  (stop [_]))

(defrecord Coendou [publish-channel
                    publication
                    publish-switch
                    metrics-agent
                    publish-interval-in-ms
                    report-channel
                    bucket-step-fn
                    cmd-state
                    window-size]
  IContext
  (get-context [_]
    (gen-context* report-channel))

  ICreateCommand
  (create-command [_ cmd-config]
    (let [cmd (create-command* cmd-config)]

      (swap! cmd-state assoc (:id cmd) cmd)
      cmd))

  IDestroyCommand
  (destroy-command [_ cmd-id]
    (let [config (get @cmd-state cmd-id)]
      (swap! cmd-state dissoc cmd-id)
      config))

  IReportSubcription
  (report-subscription [_]
    (let [ch (a/chan (a/sliding-buffer 100))]
      (a/sub publication :buckets ch)
      ch))

  ICoendouLifecycle
  (start [component]
    (let [publish-switch (atom true)]
      (reporting-loop {:switch publish-switch
                       :input-channel report-channel
                       :output-channel publish-channel
                       :cmd-state cmd-state
                       :window-size window-size
                       :bucket-step-fn (or bucket-step-fn default-bucket-step-fn)})

      (assoc component
        :report-channel report-channel
        :publish-switch  publish-switch
        :publish-channel publish-channel
        :publication     (a/pub publish-channel :topic)
        :metrics-agent   metrics-agent)))

  (stop [component]
    (when publish-switch
      (reset! publish-switch false))
    (when publish-channel
      (a/close! publish-channel))

    (when report-channel
      (a/close! report-channel))

    (assoc component
      :publish-switch nil
      :publication nil
      :publish-channel nil
      :metrics-agent nil
      :report-channel nil)))

(defn init-coendou []
  (map->Coendou {:publish-interval-in-ms 1000
                 :publish-channel (a/chan 10)
                 :report-channel (a/chan 100 (partition-by default-bucket-step-fn))
                 :window-size 120
                 :cmd-state (atom {})}))

(defn execute-command
  "Blocks and starts measuring within the current context

  Returns a promise channel"
  [dispatch-type cmd & args]
  (let [c (a/promise-chan)]
    (a/take! (-execute-command (get-context dispatch-type) cmd args)
             (fn [[t v :as value]]
               (when-not v
                 ;; A command cannot return nil
                 (println "got nil for  value" t value "for cmd" cmd)
                 )
               (a/put! c v)))
    c))

(def deref-promise-command
  (create-command*
   {:name "deref"
    :async-fn (fn [ctx args]
                (first args))
    :timeout-ms nil}))


;; REVIEW Consider generating a new buffer that will return the same result over and over

;; Requirements, needs to
;; See for examples here https://github.com/clojure/core.async/blob/master/src/main/clojure/clojure/core/async/impl/buffers.clj
(defn promise-command
  "Doesn't block unless the execute commands blocks (e.g. a full buffer) or when the result is deref-ed

  Runs the command in the background and only starts measuring within the current context when being deref-ed

  Returns a channel containing a promise channel"
  [dispatch-type cmd & args]
  (let [ctx (get-context dispatch-type)
        result-prom (a/promise-chan)
        ext-prom (a/promise-chan)
        lock (a/chan 1)]
    ;; Start command execution
    (go
      ;; REVIEW Maybe measure latency in insertion time?
      (let [[type res :as result] (a/<! (-execute-command ctx cmd args))]
        (a/>! result-prom result)

        ;; Unlock (if still locked)
        (a/poll! lock)))

    ;; Start measuring from the moment the command is deref-ed
    (a/go
      (a/>! lock ext-prom)

      ;; Second time will block, until consumer takes
      ;; this will also trigger the measurement process
      (a/>! lock ext-prom)

      (let [cmd-name (:name cmd)
            [type v] (<! (-execute-command (update ctx :prefix (fnil conj []) cmd-name)
                                           deref-promise-command
                                           [result-prom]))]
        (>! ext-prom v)))

    ;; TODO Close channel so there is no way to interrupt the above process
    #_(go
      (a/close! lock))

    ;; REVIEW can we return a promise that has some trigger on read?
    lock))

(comment
  (let [promise-ch (a/<!! (promise-command cmd))]
    (Thread/sleep 5000)
    (a/<!! promise-ch)))

(defn queue-command
  "Doesn't block unless the execute command blocks (e.g. a full buffer)

  Runs the command in the background and doesn't measure withing the current context

  Returns a closed channel"
  [dispatch-type cmd & args]
  ;; REVIEW Maybe measure latency in insertion time?
  (let [c (a/chan 1)]
    (-execute-command (get-context dispatch-type) cmd args)
    (a/put! c :started)
    c))
