(ns comms.admin
  (:require [comms.util :refer [as-properties]])
  (:import [org.apache.kafka.clients.admin AdminClient NewTopic]))


(defn create-client [properties]
  (AdminClient/create (as-properties properties)))


(defn- map->topic [{:keys [name partitions replication-factor]
                    :or {partitions 1 replication-factor 1}}]
  (NewTopic. name partitions replication-factor))


(defn create-topics [client topics]
  (.createTopics client (mapv map->topic topics)))


(defn create-topic [client topic]
  (create-topics client [topic]))


(defn delete-topics [client topics]
  (.values (.deleteTopics client topics)))


(defn delete-topic [client topic]
  (get (delete-topics client [topic]) topic))


(defn list-topics [client]
  @(.names (.listTopics client)))


(defn- node [n]
  {:has-rack (.hasRack n)
   :id (.id n)
   :is-empty (.isEmpty n)
   :port (.port n)
   :rack (.rack n)})


(defn- in-sync-replicas [is]
  (mapv node is))


(defn- partitions [ps]
  (mapv
   (fn [partition]
     {:in-sync-replicas (in-sync-replicas (.isr partition)) 
      :leader (node (.leader partition))
      :id (.partition partition)
      :replicas (mapv node (.replicas partition))})
   ps))


(defn describe-topics [client topics]
  (reduce
   (fn [acc [name value]]
     (assoc acc name
            {:name (.name value)
             :partitions (partitions (.partitions value))}))
   {}
   @(.all (.describeTopics client topics))))


(defn describe-topic [client topic]
  (get (describe-topics client [topic]) topic))


(defn describe-cluster [client]
  (let [description (.describeCluster client)]
    {:cluster-id @(.clusterId description)
     :controller (node @(.controller description))
     :nodes (mapv node @(.nodes description))}))


